// Exercises the FFT class

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
You should have received a copy of the GNU General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include<complex.h>
#include<mMatrix.h>
#include<instance.h>
#include<fft.h>

typedef mVector<double,allocator<double> > dbl_vctr;

typedef mVector<double_complex,allocator<double_complex> > cmplx_vctr;

double (*np)(const double_complex&)=norm;

double test_fft(cmplx_vctr &f,cmplx_vctr &fft) {
// Computes the L2-norm of the error between the FFT and the discrete FT.
  int n=f.size();
  double err=0;
  double_complex ftmp,arg;
  for(int j=1;j<=n;j++) {   // accumulate the FFT the hard way
    ftmp=double_complex(0,0);
    for(int k=1;k<=n;k++) {
      arg=double_complex(0,2*M_PI*(j-1)*(k-1)/n);
      ftmp+=f(k)*exp(arg);
    }
    ftmp/=sqrt(n);
    err+=norm(ftmp-fft(j));
  }
  err/=accumulate(fft,0.0,np);
  return sqrt(err);
}

double test_fft(dbl_vctr &f,cmplx_vctr &fft) {
// Computes the L2-norm of the error between the FFT and the discrete FT.
  int n=f.size();
  cmplx_vctr v(n),vfft(n);
  for(int j=1;j<=n;j++) v(j)=f(j);
  vfft(1)=real(fft(1));
  vfft(n/2+1)=imag(fft(1));
  for(int j=1;j<n/2;j++) {
    vfft(j+1)=fft(j+1);
    vfft(n-j+1)=conj(fft(j+1));
  }
  return test_fft(v,vfft);
}

double test_sin(dbl_vctr &f,dbl_vctr &fft) {
// Computes the L2-norm of the error between the FST and the discrete FT.
  int n=f.size()+1;
  double err=0;
  double ftmp,arg;
  for(int j=1;j<n;j++) {   // accumulate the FFT the hard way
    ftmp=0;
    for(int k=1;k<n;k++) {
      arg=M_PI*j*k/n;
      ftmp+=f(k)*sin(arg);
    }
    ftmp*=sqrt(2.0/n);
    err+=norm(ftmp-fft(j));
  }
  err/=accumulate(fft,0.0,np);
  return sqrt(err);
}

double test_cos(dbl_vctr &f,dbl_vctr &fft) {
// Computes the L2-norm of the error between the FST and the discrete FT.
  int n=f.size()-1;
  double err=0;
  double ftmp;
  int s=1;
  for(int k=0;k<=n;k++,s*=-1) {   // accumulate the FFT the hard way
    ftmp=0.5*(f(1)+s*f(n+1));
    for(int j=1;j<n;j++) ftmp+=f(j+1)*cos(k*j*M_PI/n);
    ftmp*=sqrt(2.0/n);
    err+=norm(ftmp-fft(k+1));
  }
  err/=accumulate(fft,0.0,np);
  return sqrt(err);
}

int main(int argc,char **argv) {
  int dim=atoi(argv[1]);
  cout << "mVector length: " << dim << "\n\n";
  srand(1);              // seed the random # generator with a known value
  double maxr=(double)0x0fffffff;
  
  FFT<double,complex<double> > a;
  
  cout << "Complex FFT: \n";
  cmplx_vctr f(dim);
  for(int j=1;j<=f.size();j++) {
    f(j)=double_complex(rand()/maxr,rand()/maxr);
  }
  cmplx_vctr g(dim);
  g=f;
  a.fft(f,1);
  double err=sqrt(accumulate(f,0.0,np))-sqrt(accumulate(g,0.0,np));
  cout << "\tTest of discrete Parsival: " << err << "\n";
  cout << "\tError of FFT from discrete FT: " << test_fft(g,f) << "\n";
  a.fft(f,-1);
  err=sqrt(accumulate(f-g,0.0,np)/accumulate(g,0.0,np));
  cout << "\tError of inverse FFT: " << err << "\n\n";

  cout << "Real FFT: \n";
  dbl_vctr r(dim);
  for(int j=1;j<=r.size();j++) {
    r(j)=rand()/maxr;
  }
  dbl_vctr s(dim);
  s=r;
  a.fft(r,1);
  err=norm(a.vc(1));
  for(int j=2;j<=a.vc.size();j++) err+=2*norm(a.vc(j));
  err=sqrt(err);
  err-=sqrt(accumulate(r,0.0,np));
  cout << "\tTest of discrete Parsival: " << err << "\n";
  cout << "\tError of FFT from discrete FT: " << test_fft(r,a.vc) << "\n";
  a.fft(r,-1);
  err=sqrt(accumulate(s-r,0.0,np)/accumulate(s,0.0,np));
  cout << "\tError of inverse FFT: " << err << "\n\n";

  cout << "Real FST: \n";
  dbl_vctr fs(dim-1);
  for(int j=1;j<=fs.size();j++) {
    fs(j)=rand()/maxr;
  }
  dbl_vctr gs(dim-1);
  gs=fs;
  a.sin(fs);
  err=sqrt(accumulate(fs,0.0,np))-sqrt(accumulate(gs,0.0,np));
  cout << "\tTest of discrete Parsival: " << err << "\n";
  cout << "\tError of FST from discrete ST: " << test_sin(gs,fs) << "\n";
  a.sin(fs);
  err=sqrt(accumulate(fs-gs,0.0,np)/accumulate(gs,0.0,np));
  cout << "\tError of inverse FFT: " << err << "\n\n";

  cout << "Real FCT: \n";
  dbl_vctr fc(dim+1);
  for(int j=1;j<=fc.size();j++) {
    fc(j)=rand()/maxr;
  }
  dbl_vctr gc(dim+1);
  gc=fc;
  a.cos(fc);
  err=sqrt(accumulate(fc,0.0,np))-sqrt(accumulate(gc,0.0,np));
  cout << "\tNo discrete Parsival: " << err << "\n";
  cout << "\tError of FCT from discrete CT: " << test_cos(gc,fc) << "\n";
  a.cos(fc);
  err=sqrt(accumulate(fc-gc,0.0,np)/accumulate(gc,0.0,np));
  cout << "\tError of inverse FFT: " << err << "\n\n";

  cout << "If any numbers except FCT discrete Parsival are nonzero,\n";
  cout << "then it indicates a problem which should be investigated.\n";

  return 0;
}
