// Tests the pseudo-inverse for overdetermined matrix problems

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
You should have received a copy of the GNU General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include<complex.h>
#include<mMatrix.h>
#include<instance.h>
#include<lin_alg.h>

int main(int argc,char **argv) {
  int ldim=atoi(argv[1]);   // number of rows
  int dim=atoi(argv[2]);    // number of columns, less than or equal to rows
  mMatrix<double_complex,allocator<double_complex> > a(ldim,dim),i(dim,dim),il(ldim,dim);
  mVector<double_complex,allocator<double_complex> > p(ldim),q(ldim),s(dim);
  mVector<double_complex,allocator<double_complex> > tvl(ldim),tv(dim);
  srand(1);              // seed the random # generator with a known value
  double maxr=(double)0x0fffffff;
  for(int r=1;r<=dim;r++) {  // set a to a random matrix, i to the identity
    for(int c=1;c<=dim;c++) {
      a(r,c)=double_complex(rand()/maxr,rand()/maxr);
      i(r,c)=0.0;
      il(r,c)=0.0;
    }
    i(r,r)=1.0;
    il(r,r)=1.0;
    p(r)=double_complex(rand()/maxr,rand()/maxr);   // and p to a random vector
  }
  for(int r=dim+1;r<=ldim;r++) {
    for(int c=1;c<=dim;c++) {
      a(r,c)=double_complex(rand()/maxr,rand()/maxr);
      il(r,c)=0.0;
    }
    p(r)=double_complex(rand()/maxr,rand()/maxr);
  }
  mMatrix<double_complex,allocator<double_complex> > b(ldim,dim);
  b=a;
  cout << "Test mMatrix is random double_complex, with "<< ldim;
  cout << " rows and " << dim << " columns\n";
  cout << "Test singular value decomposition, using the sup norm with abs: \n";
  mMatrix<double_complex,allocator<double_complex> > u(ldim,dim),v(dim,dim);
  mMatrix<double_complex,allocator<double_complex> > tl(ldim,dim),t(dim,dim);
  u=il;
  v=i;
  mVector<double,allocator<double> > sv1(dim);
  svd(b,u,sv1,v,0.0);
  mMatrix<double_complex,allocator<double_complex> > ua(ldim,dim),va(dim,dim);
  ua=u;
  va=v;
  ua.transpose();
  double_complex (*cp)(const double_complex&)=conj;
  transform(ua,ua,cp);
  va.transpose();
  transform(va,va,cp);
  cout << "Orthogonality, deviation from identity: ";
  cout << "u: " << sup(i-dot(ua,u,t)) << '\t';
  cout << "v: " << sup(i-dot(va,v,t)) << '\n';
  cout << "Ratio, smallest to largest SV: " << sv1(dim)/sv1(1) << '\n';
  for(int r=1;r<=dim;r++) va.row(r)*=(double_complex)sv1(r);
  cout << "Reconstruction, deviation from original: ";
  cout << sup(a-dot(u,va,tl)) << '\n';

  cout << "Nonsingular inverse by pseudo_inverse():\n";
  b=a;         // reset b, u and v
  u=il;
  v=i;
  pseudo_inverse(b,u,sv1,v,0.0);
  cout << "Deviation from the inverse: " << sup(i-dot(b,a,t)) << '\n';

  cout << "Singular inverse by pseudo_inverse():\n";
  b.transpose();
  b=a;         // reset b, u and v
  u=il;
  v=i;
  b.col(dim)=b.col(1);      // set one column equal to another
  mMatrix<double_complex,allocator<double_complex> > c(ldim,dim);
  c=b;
  pseudo_inverse(b,u,sv1,v,0.0);    // invert and test
  ua.transpose();
  ua=u;
  ua.transpose();
  transform(ua,ua,cp);
  va.transpose();
  va=v;
  va.transpose();
  transform(va,va,cp);
  cout << "Orthogonality, deviation from identity: ";
  cout << "u: " << sup(i-dot(ua,u,t)) << '\t';
  cout << "v: " << sup(i-dot(va,v,t)) << '\n';
  cout << "Ratio, smallest to largest SV: " << sv1(dim)/sv1(1) << '\n';
  for(int r=1;r<=dim;r++) va.row(r)*=(double_complex)sv1(r);
  cout << "Reconstruction, deviation from original: ";
  cout << sup(c-dot(u,va,tl)) << '\n';
//  q=u.col(dim);     // the left singular vector
assign(u.col(dim),q);
  transform(q,q,cp);
//  s=v(dim);     // the right singular vector
assign(v.col(dim),s);
  transform(s,s,cp);
  cout << "Test singular vectors: left: " << sup(dot(q,c,tv)) << "\t";
  cout << "right: " << sup(dot(c,s,tvl)) << "\n";
  double (*np)(const double_complex&)=norm;
  cout << "Normalization: " << 1.0-accumulate(q,0.0,np) << "\n";
// we can only solve the problem where p is restriced to the space 
// spanned by the nonsingular columns of u, so that
  q=(double_complex)0.0;
  for(int r=1;r<dim;r++) q+=sum(ua.row(r)*p)*u.col(r);
  dot(b,q,s);         // the solution to the inverse problem
  cout << "Deviation of solution: " << sup(q-dot(c,s,tvl)) << '\n';

  cout << "The accuracy of these routines is indicated by the size of the ";
  cout << "numbers reported above.\n";
  return 0;
}
