// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

PhonedevService::PhonedevService() :
Thread(NULL, keythreads.priService(), phivr.getStack()), Mutex()
{
	long opt;

	first = last = NULL;
	FD_ZERO(&connect);
	::pipe(iosync);
	hiwater = iosync[0] + 1;

	opt = fcntl(iosync[0], F_GETFL);
	fcntl(iosync[0], F_SETFL, opt | O_NDELAY);
}

PhonedevService::~PhonedevService()
{
	Update(0);
	Terminate();

	while(first)
		delete first;
}

void PhonedevService::Attach(PhonedevTrunk *trunk)
{
	int dev;

	EnterMutex();
	if(last)
		last->next = trunk;

	trunk->prev = last;
	last = trunk;
	dev = trunk->dev;
	FD_SET(dev, &connect);
	if(dev >= hiwater)
		hiwater = ++dev;

	if(!first)
	{
		first = trunk;
		LeaveMutex();
		Start();
	}
	else
	{
		LeaveMutex();
		Update();
	}
	++count;
}

void PhonedevService::Detach(PhonedevTrunk *trunk)
{
	EnterMutex();
	FD_CLR(trunk->dev, &connect);
	if(trunk == first && trunk == last)
	{
		first = last = NULL;
		LeaveMutex();
		return;
	}
	if(trunk->prev)
		trunk->prev->next = trunk->next;

	if(trunk->next)
		trunk->next->prev = trunk->prev;

	if(trunk == first)
		first = trunk->next;

	if(trunk == last)
		last = trunk->prev;

	LeaveMutex();
	Update();
	--count;
}		

void PhonedevService::Update(unsigned char flag)
{
	::write(iosync[1], (char *)&flag, 1);
}

void PhonedevService::Run(void)
{
	TrunkEvent event;
	struct timeval timeout, *tvp;
	timeout_t timer, expires;
	PhonedevTrunk *trunk;
	fd_set err;
	fd_set inp;
	unsigned char buf;
	int dev;
	char tmp[33];

	FD_ZERO(&err);

	setCancel(THREAD_CANCEL_DEFERRED);
	for(;;)
	{
		timer = ~0;
		while(1 == ::read(iosync[0], (char *)&buf, 1))
		{
			if(buf && buf != 0xff)
			{ 
				debug->DebugService(driver->getTrunkPort(buf - 1), "notify");
				phivr.Notify(buf);
				continue;
			}
			if(buf)
			{
				debug->DebugService(NULL, "updating");
				continue;
			}
			debug->DebugService(NULL, "exiting");
			setCancel(THREAD_CANCEL_IMMEDIATE);
			Sleep(~0);
			Exit();
		}

		Yield();
		EnterMutex();
		trunk = first;
		while(trunk)
		{
			dev = trunk->dev;

			if(FD_ISSET(dev, &err))
			{
				debug->DebugService(trunk, "event");
				trunk->getEvents();
			}

retry:
			expires = trunk->getTimer();
			if(expires > 0)
				if(expires < timer)
					timer = expires;

			if(!expires)
			{
				debug->DebugService(trunk, "expires");
				event.id = TRUNK_TIMER_EXPIRED;
				trunk->endTimer();
				trunk->postEvent(&event);
				goto retry;
			}

			trunk = trunk->next;
		}
		LeaveMutex();
		memcpy(&err, &connect, sizeof(err));
		FD_ZERO(&inp);
		FD_SET(iosync[0], &inp);
		if(timer == ~0)
			tvp = NULL;
		else
		{
			tvp = &timeout;
			timeout.tv_sec = timer / 1000;
			timeout.tv_usec = (timer % 1000) * 1000;
		}
		sprintf(tmp, "delay %d", timer);
		debug->DebugService(NULL, tmp);
		select(hiwater, &inp, NULL, &err, tvp);
		debug->DebugService(NULL, "wakeup");
	}
}

