/* Copyright (C) 1997 Aladdin Enterprises.  All rights reserved.
  
  This file is part of GNU Ghostscript.
  
  GNU Ghostscript is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility to
  anyone for the consequences of using it or for whether it serves any
  particular purpose or works at all, unless he says so in writing.  Refer to
  the GNU General Public License for full details.
  
  Everyone is granted permission to copy, modify and redistribute GNU
  Ghostscript, but only under the conditions described in the GNU General
  Public License.  A copy of this license is supposed to have been given to
  you along with GNU Ghostscript so you can know your rights and
  responsibilities.  It should be in a file named COPYING.  Among other
  things, the copyright notice and this notice must be preserved on all
  copies.
  
  Aladdin Enterprises is not affiliated with the Free Software Foundation or
  the GNU Project.  GNU Ghostscript, as distributed by Aladdin Enterprises,
  does not depend on any other GNU software.

  However Display Ghostscript System depends on GLib. */

/* $Id: gsdpsx.c,v 1.50 1999/09/17 19:06:56 masata-y Exp $ */
/* X Windows Display PostScript additions for Ghostscript library */


#define DUMP_UNHANDLED_MESSAGES 1

#include <glib.h>		/* For Hash tables */
typedef GSList GSet;

#include "math_.h"
#include "gx.h"
#include "gserrors.h"
#include "gsmatrix.h"		/* for gscoord.h */
#include "gscoord.h"
#include "gspaint.h"
#include "gspath.h"
#include "gsexit.h"
#include "gxfixed.h"
#include "gzstate.h"
#include "iref.h"
#include "imain.h"
#include "iminst.h"
#include "ialloc.h"

/* for the DPS/X11 streams */
#include "stream.h"
#include "gxiodev.h"

/* the X people can't make their headers define what they need automatically... */
#define NEED_REPLIES
#define NEED_EVENTS
#include "x_.h"                 /* standard X include files */
#include "string_.h"
#include "gxdevice.h"
#include "X11/Xproto.h"
#include "dpsNXprops.h" /* has the property definitions */
#include "DPS/XDPSproto.h"
#include "DPSCAPproto.h"
#include "DPSCAP.h"
#include "DPS/dpsNXargs.h"
#include "DPS/XDPS.h"

#include "gdevx.h"
extern gx_device_X gs_x11_device;

#include "gsdpsx.h"

/* network headers for the agent */
#include <sys/socket.h>
#include <netinet/in.h> /* for ntohs */
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>

/* required by fork based server */
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

/* headers for select(2) */
#include <sys/time.h>
#include <errno.h>
private int DGS_DEBUG = 0;

/* Supported connection types */
enum conn_type { UNIXCONN = XDPSNX_TRANS_UNIX, TCPCONN =  XDPSNX_TRANS_TCP};

/* For UNIX CONN */
#include <sys/stat.h>
#include <sys/un.h>

#ifndef CSDPS_UNIX_PATH
#ifdef hpux
#define CSDPS_UNIX_DIR      "/usr/spool/sockets/DPSNX"
#define CSDPS_UNIX_PATH     "/usr/spool/sockets/DPSNX/"
#else  /* !hpux */
#define CSDPS_UNIX_DIR      "/tmp/.DPSNX-unix"
#define CSDPS_UNIX_PATH     "/tmp/.DPSNX-unix/AGENT"
#endif /* hpux */
#endif /* CSDPS_UNIX_DIR */

/* Constants */
#define BUFLEN 65536
#define HOSTNAME_LEN 64

/* --- private data --- */
typedef struct gsdpsx_conn_buf_s 
{
    unsigned char *buf; 
    unsigned char *curpos; 
    unsigned int   len; /* if we need to leave messages here, the remaining length */
    unsigned char *inputstart; /* these are for when we get an input message in here */
    unsigned char *inputend;
    int          isfrag; /* first item in the buffer is a partial fragment */
} gsdpsx_conn_buf;

struct ctx_init_parameter 
{
  CONTEXTXID cxid;
  Drawable drawable;
  int offset_x;
  int offset_y;
};
private void ctx_init_parameter_initialize(struct ctx_init_parameter  * parameter);
private int ctx_init_parameter_install(struct ctx_init_parameter  * parameter);

typedef struct gsdpsx_conn_info_s 
{
    Window agent_win;
    Window       clientWindow; 
    unsigned int  serial;
    CARD16       displayNum; /* don't know, but we get it... */
    CARD16       screen;
    Atom resume_atom;
    gx_device_X * primary_device;

    int          sock;
    unsigned char byteorder;
    enum conn_type conn_type;
    gsdpsx_conn_buf * buf;

    int context_state_index_primary;
    int context_state_index_last;
  
    GHashTable * cpsid_to_cxid_table;
    GHashTable * cxid_to_cpsid_table;
    GSet * pause_cxids_set;
    GSet * kill_cpsids_set;
    GHashTable * cpsid_to_mask_table;
    GHashTable * cpsid_to_nextMask_table;
    struct ctx_init_parameter ctx_init_parameter;
} gsdpsx_conn_info;
private gsdpsx_conn_info conn_info;

private Atom billboard_atom;
private int billboard_atom_needs_cleanup;

private Display *shared_dpy; /*! NOT GOOD NAME */
private unsigned char shared_byteorder;
private Window server_win_unix;
private Window server_win_tcp;

private int server_socket_unix;	
private int server_socket_tcp;	
private struct sockaddr_un server_sockaddr_unix; /*! NOT GOOD */
private struct sockaddr_in server_sockaddr_tcp; /*! NOT GOOD */

private int server_agent_counter = 0;
private GSet * server_agent_pids = NULL;

/* --- context init parameter  --- */
private GSet * g_set_add(GSet * set, gpointer data);
private gpointer g_set_member (GSet *set, gpointer   data);
private GSet * g_set_remove(GSet * set, gpointer data);
private void g_set_foreach (GSet *set, GFunc func, gpointer   user_data);
private guint g_set_count  (GSet * set);

/* declaration secret functions in zcontext.c */
int gsdpsx_get_current_context_index(void);
void gsdpsx_request_context_index(int index);
void gsdpsx_clear_newest_context(void);
int gsdpsx_get_newest_context_index(void);
int gsdpsx_count_ready_context(void);
void gsdpsx_set_primary_context_index(int index);


/* --- forward prototypes --- */

/* yield_op */
private const int yield_op_length = 12;
private const byte yield_op_data [] = 
{129, 1, 12, 0, 131, 0, 255, 255, 115, 1, 0, 0,};

/* fork op */
private const int fork_op_length = 50;
private const byte fork_op_data [] = 
{
  129, 3,  50,   
  0, 131,   0, 255, 255,  62,   1,  0,
  0, 137,   0,   2,   0,  24,   0,  0,   
  0, 131,   0, 255, 255,  48,   1,  0,
  0,   5,   0,   6,   0,  40,   0,  0,   
  0, 131,   0, 255, 255,  79,   1,  0,
  0,  37, 115, 116, 100, 105, 110    
};

/* detach op */
private const int detach_op_length = 20;
private const byte detach_op_data [] =
{
  129,   2,  20,
  0  , 131,   0,  255,  255, 22, 1,0, /* currentcontext */
  0  , 131,   0,  255,  255, 36, 1,0, /* detach */
  0
};
enum scheduling_action 
{
  SCHED_ACT_NOTHING, /* Forward data from socket to the gs interpreter */
  SCHED_ACT_FORK,    /* Fork request */
  SCHED_ACT_YIELD,   /* Context switching request  */
  SCHED_ACT_DETACH   /* Context killing request */
};

/* message handlers */
/* returns the number of bytes message used.  0 if it is just a fragment. */
/* returns negative on a different error. */

/* DPSCAP  */
private int dpscap_ConnSetup(gsdpsx_conn_info *conn, 
			     unsigned char *pmsg, int remaining);

private int dpscap_Notify    (gsdpsx_conn_info *conn, 
			      unsigned char *pmsg, int remaining,
			      enum scheduling_action * sched_act);
private Bool dpscap_Notify_pause_check(gsdpsx_conn_info *conn,
				       int cxid,
				       int sync);

private int dpscap_FlushAgent(gsdpsx_conn_info *conn, 
			      unsigned char *pmsg, int remaining);
private int dpscap_SetArg    (gsdpsx_conn_info *conn, 
			      unsigned char *pmsg, int remaining);

/* DPS  */
private int dps_CreateContext(gsdpsx_conn_info *conn, 
			      unsigned char *pmsg, int remaining,
			      enum scheduling_action * sched_act);
private void dps_CreateContext_reply(gsdpsx_conn_info *conn, int cpsid);

private int dps_SetStatusMask(gsdpsx_conn_info *conn, 
			      unsigned char *pmsg, int remaining);
private int dps_NotifyContext(gsdpsx_conn_info *conn, 
			      unsigned char *pmsg, int remaining,
			      enum scheduling_action * sched_act);
private int dps_DestroySpace (gsdpsx_conn_info *conn,
			      unsigned char *pmsg, int remaining);
private int dps_GiveInput    (gsdpsx_conn_info *conn,
			      unsigned char *pmsg, int remaining,
			      enum scheduling_action * sched_act);
private int dps_Init         (gsdpsx_conn_info *conn,
			      unsigned char *pmsg, int remaining);
private int dps_CreateSpace   (gsdpsx_conn_info *conn,
			       unsigned char *pmsg, int remaining);
private int dps_GetStatus    (gsdpsx_conn_info *conn,
			      unsigned char *pmsg, int remaining);
private int dps_Reset        (gsdpsx_conn_info *conn,
			      unsigned char *pmsg, int remaining);
private int dps_CreateContextFromID(gsdpsx_conn_info *conn,
				    unsigned char *pmsg, int remaining);
private int dps_XIDFromContext(gsdpsx_conn_info *conn,
			       unsigned char *pmsg, int remaining);
private int dps_ContextFromXID(gsdpsx_conn_info *conn,
			       unsigned char *pmsg, int remaining);
private int dps_CreateSecureContext(gsdpsx_conn_info *conn,
				    unsigned char *pmsg, int remaining);
private int dps_NotifyWhenReady(gsdpsx_conn_info *conn,
				unsigned char *pmsg, int remaining);

/* DPSX */
private int dpsx_ChangeGC(gsdpsx_conn_info *conn,
				unsigned char *pmsg, int remaining);

/* CONNection setting up functions */
private void conn_sock_initialize(void);
private void conn_sock_finalize(void * sockaddr, enum conn_type type);

private int conn_server_create(void * sockaddr, enum conn_type type);
private int conn_server_bind(int server_socket, 
			     void * sockaddr,
			     int base_port,
			     enum conn_type type);
private void conn_server_close(int server_socket);
private int conn_base_port_find(const char * args);
private int conn_hostname_get(char * hostname, enum conn_type type);

private int conn_atom_setup(Display *shared_dpy, 
			    Window win_unix, int port_unix,
			    Window win_tcp, int port_tcp);
private void conn_atom_cleanup(void);
private int conn_insock_accept(int server_socket, enum conn_type type);
private void conn_insock_close(int insock);

/* PROCess control functsions (parent) */
private int proc_agent_start(int in_sock, 
			     int server_socket_unix, 
			     int server_socket_tcp, 
			     gs_state * pgs, gs_memory_t *mem,
			     int agent_number, enum conn_type type);
private void proc_server_wait_agent(int);

private void proc_agent_finalize(int); 
private void proc_server_finalize(int); 

private void proc_server_setup_signal(void);
private void proc_agent_setup_signal(void);

private void proc_server_control(int);
private void proc_agent_control(int);

/* buf */
private gsdpsx_conn_buf * buf_init(void);
private int buf_recv(gsdpsx_conn_buf * buf, gsdpsx_conn_info * conn);
private void buf_input_set(gsdpsx_conn_buf * buf, 
			   unsigned char * start, int length);
private int buf_input_length(gsdpsx_conn_buf * buf);
private void buf_input_forward(gsdpsx_conn_buf * buf, stream_cursor_write *pw);
private void buf_pack(gsdpsx_conn_buf * buf, unsigned char *cur, int len);
private void buf_move_cursor(gsdpsx_conn_buf * buf, unsigned char *cur, int len);
private void buf_reset(gsdpsx_conn_buf * buf);
private void buf_input_sanity_check(gsdpsx_conn_buf * buf);
private void buf_dump(gsdpsx_conn_buf * buf, int length, unsigned char op);
private void buf_finalize(gsdpsx_conn_buf * buf);

/* args */
private const char * args_key = "DGS_ARGS";
private const char * args_get(void);
private int args_get_port(const char * args);
private int args_get_int(const char * args, const char * key);
private int args_get_match(const char * args, const char * key);
private int args_get_debug(const char * args);

/* status */
private int status_send_if_masked(gsdpsx_conn_info *conn,
				  int cpsid,
				  int status,
				  int steady1_or_next0);

/* utility functions */
private const char * util_x_ps_request_enum2str(int key);
private const char * util_x_cap_request_enum2str(int key);
private const char * util_signal_int2str(int sig);
private const char * util_status_enum2str(int key);
private const char * util_conn_type_enum2str(enum conn_type type);
private void util_print_pid(int sec);
private int util_byteorder_check(gsdpsx_conn_info *conn);

/* ------ DPS/X11 streams ------ */
private void gsdpsx_std_init(void);

/* ------ Agent state ------ */

/* --- Starts the X DPS agent --- */
int
gs_startXdpsagent(gs_state *pgs, gs_memory_t *mem)
{
    int server_port_unix, server_port_tcp;
    int base_port;
    fd_set rfds;
    int max_port;
    int result;
    int indian = 1;
    const char * dgs_args = args_get();

    DGS_DEBUG=args_get_debug(dgs_args);
    if (DGS_DEBUG < 0)
      DGS_DEBUG = 0;
    else if (DGS_DEBUG < 0)
      dprintf("Enter debug mode\n");
    
    if (args_get_match(dgs_args, "help"))
      {
	dprintf("Usage: dpsnx.agent [--help] [--debug=N] [--version] [tcp/PORT|unix/PORT]\n");
	dprintf("[--help]: Print this messages\n");
	dprintf("[--debug=N]: Wait N sec before forking dgs agent"
		"and enable to print debug messages.\n");
	dprintf("[tcp/PORT|unix/PORT]: Open port number PORT to the clients.\n");
	gs_exit(0);
      }
    else if (args_get_match(dgs_args, "version"))
      {
	dprintf2("GNU Display Ghostscript System %s based on Ghostscript %s\n", 
		 VERSION,
		 GS_VERSION);
	dprintf("Copyright (C) 1997 Aladdin Enterprises.  All rights reserved.\n");
	dprintf("You may redistribute copies of GNU Display Ghostscript System\n"
		"under the terms of the GNU General Public License.\n"
		"For more information about these matters, see the files named COPYING.\n");
	gs_exit(0);
      }
    
    conn_sock_initialize();

    /** Check server side byteorder, see dps/cxlibext.c */
    if (*(char *) &indian)
        shared_byteorder = 'l';
    else
        shared_byteorder = 'B';

    /** open the display, this will last the entire session and be used for
       every window used */
    shared_dpy = XOpenDisplay(NULL);
    if (!shared_dpy) {
	fprintf(stderr, "Cannot open X Display(server): %s\nABORTING!\n", 
	        XDisplayName(NULL));
        return -1;
    }

    /** create us a socket to listen on */
    server_socket_unix = conn_server_create(&server_sockaddr_unix, UNIXCONN);
    if (server_socket_unix == -1) goto close_display;

    server_socket_tcp = conn_server_create(&server_sockaddr_tcp, TCPCONN);
    if (server_socket_tcp == -1) goto close_socket_unix;
    
    /** find an open port */
    base_port = conn_base_port_find(dgs_args); 
    server_port_unix = conn_server_bind(server_socket_unix, 
					&server_sockaddr_unix, 
					base_port,
					UNIXCONN);
    if (server_port_unix == -1)
      goto finalize_sock_unix;

    server_port_tcp = conn_server_bind(server_socket_tcp,
				       &server_sockaddr_tcp,
				       base_port,
				       TCPCONN);

    if (server_port_tcp == -1)
      goto finalize_sock_tcp;

    /** a base window */
    /* UNIXCONN */
    server_win_unix 
      = XCreateSimpleWindow(shared_dpy, 
			    DefaultRootWindow(shared_dpy),
			    0, 0, 1, 1, /* min X window size */
			    0, /* no border either */
			    BlackPixel(shared_dpy, DefaultScreen(shared_dpy)),
			    WhitePixel(shared_dpy, DefaultScreen(shared_dpy)));
    if (!server_win_unix) 
      {
	fprintf(stderr, 
		"Unable to create main comm window for UNIXCONN.\n");
	goto close_socket_tcp;
      }
    /* we want basically no input, but that does give us Client messages */
    XSelectInput(shared_dpy, server_win_unix, NoEventMask);

    /* TCPCONN */
    server_win_tcp
      = XCreateSimpleWindow(shared_dpy, 
			    DefaultRootWindow(shared_dpy),
			    0, 0, 1, 1, /* min X window size */
			    0, /* no border either */
			    BlackPixel(shared_dpy, DefaultScreen(shared_dpy)),
			    WhitePixel(shared_dpy, DefaultScreen(shared_dpy)));
    if (!server_win_tcp) {
	fprintf(stderr,
		"Unable to create main comm window for TCPCONN.\n");
	goto destory_win_unix;
    }
    /* we want basically no input, but that does give us Client messages */
    XSelectInput(shared_dpy, server_win_tcp, NoEventMask);
 
    /** that we are ready to accept connections, we must inform the server */
    result = conn_atom_setup(shared_dpy, 
			     server_win_unix, 
			     base_port +server_port_unix,
			     server_win_tcp, 
			     base_port +server_port_tcp);
    if (-1 == result)
      goto destory_win_tcp;
    billboard_atom_needs_cleanup = 1;
    /* flush all remaining X events */
    XSync(shared_dpy, False);

    /******* alright, its time to really accept a connection and get to work ********/
    /* this code will all have to be rewritten with multiple contexts in mind */

    while (1)
      {
	int in_sock;
	enum conn_type type;

	result = listen(server_socket_unix, 5);
	if (result < 0)
	  {
	    perror("[UNIX]listen: ");
	    goto atom_cleanup;
	  }
	
	result = listen(server_socket_tcp, 5);
	if (result < 0)
	  {
	    perror("[TCP]listen: ");
	    goto atom_cleanup;
	  }
	max_port = (server_socket_unix<server_socket_tcp)?
	  server_socket_tcp:server_socket_unix;
    
	FD_ZERO(&rfds);
	FD_SET(server_socket_unix, &rfds);
	FD_SET(server_socket_tcp, &rfds);
	proc_server_setup_signal();
	result = select(max_port+1, &rfds, NULL, NULL, NULL);
	if (-1 == result)
	  {
	    if (errno != EINTR)
	      {
		perror("select: ");
		goto atom_cleanup;
	      }
	    else
		continue ;	/* SIGCHLD */
	  }

	if (FD_ISSET(server_socket_unix, &rfds))
	  {
	    in_sock = conn_insock_accept(server_socket_unix, 
					 UNIXCONN);
	    type = UNIXCONN;
	    dprintf2("Connection accepted on socket number %d via %s\n", 
		     in_sock, "UNIXCONN");
	  }
	else if (FD_ISSET(server_socket_tcp, &rfds))
	  {
	    in_sock = conn_insock_accept(server_socket_tcp, 
					 TCPCONN);
	    type = TCPCONN;
	    dprintf2("Connection accepted on socket number %d via %s\n", 
		     in_sock, "TCPCONN");
	  }

	if (in_sock == -1) 
	  goto atom_cleanup;

	/** setup basic connection info and run */
	result = proc_agent_start(in_sock, 
				  server_socket_unix, 
				  server_socket_tcp, 
				  pgs, 
				  mem,
				  ++server_agent_counter,
				  type);
	if (-1 == result)
	  goto atom_cleanup;
      }
 atom_cleanup:
    conn_atom_cleanup();
 destory_win_tcp:
    XDestroyWindow(shared_dpy, server_win_tcp);
 destory_win_unix:
    XDestroyWindow(shared_dpy, server_win_unix);
 finalize_sock_tcp:
    conn_sock_finalize(&server_sockaddr_tcp, TCPCONN);
 finalize_sock_unix:
    conn_sock_finalize(&server_sockaddr_unix, UNIXCONN);
 close_socket_tcp:
    conn_server_close(server_socket_tcp);
 close_socket_unix:
    conn_server_close(server_socket_unix);
 close_display:
    XCloseDisplay(shared_dpy);    
    return -1; /* if we got here, it is an error */
}


/* --- (proc_) process switch functsions --- */
private Window
proc_agent_setup_window(Display ** dpy, 
			int agent_number);

#define RETURN(code) {gs_exit(code); return code;}
private int
proc_agent_start(int in_sock, 
		 int server_socket_unix, 
		 int server_socket_tcp, 
		 gs_state * pgs, gs_memory_t *mem,
		 int agent_number,
		 enum conn_type type)
{
  int pid;
  int exit_code;
  ref error_object;
  int setup_result;

  if ((pid = fork()) < 0) {
    perror ("fork");
    return -1;
  }

  if (pid == 0) 
    {
      /** Child */
      const char * run_string = "(%stdin) run  ";
      int len;				     /* message length */

      util_print_pid(DGS_DEBUG);

      /** Shutdown some server resources */
      billboard_atom_needs_cleanup = 0;	     /* atom_cleanup:  */
      server_win_tcp = 0;		     /* destory_win_tcp: */
      server_win_unix = 0;		     /* destory_win_unix: */
      conn_server_close(server_socket_tcp);  /* close_socket_tcp: */
      conn_server_close(server_socket_unix); /* close_socket_unix: */
      /* close_display: No need */

      conn_info.conn_type = type;
      conn_info.sock = in_sock;
      conn_info.buf = 0; /* value is checked in proc_agent_finalize */
      proc_agent_setup_signal();

      /* CONTEXT HACK */
      conn_info.cpsid_to_cxid_table = g_hash_table_new(NULL, NULL);
      conn_info.cxid_to_cpsid_table = g_hash_table_new(NULL, NULL);
      conn_info.cpsid_to_mask_table = g_hash_table_new(NULL, NULL);
      conn_info.cpsid_to_nextMask_table = g_hash_table_new(NULL, NULL);
      conn_info.context_state_index_primary = gsdpsx_get_current_context_index();
      gsdpsx_set_primary_context_index(conn_info.context_state_index_primary);
      conn_info.context_state_index_last = conn_info.context_state_index_primary;
      ctx_init_parameter_initialize(&(conn_info.ctx_init_parameter));
      conn_info.primary_device = NULL;
      conn_info.pause_cxids_set = NULL;
      conn_info.kill_cpsids_set = NULL;

      /** Agent own window */
      conn_info.agent_win = proc_agent_setup_window(&shared_dpy, agent_number); 
      if (-1 == conn_info.agent_win)
	{
	  conn_insock_close(conn_info.sock);
	  RETURN(-1);
	}
      conn_info.resume_atom = XInternAtom(shared_dpy, DPSCAP_TYPE_RESUME, FALSE);

      conn_info.serial = 1;
      
      /* Set up buffer */
      conn_info.buf = buf_init();
      if (!(conn_info.buf))
	{
	  proc_agent_finalize(-1);
	  RETURN(-1);
	}

      /* get the connection setup request */
    recv_again:
      len = buf_recv(conn_info.buf, &conn_info);
      if (len == -1)
	{
	  if (errno !=  EINTR)
	    {
	      perror("recv ");
	      proc_agent_finalize(-1);
	      RETURN(-1);
	    }
	  else
	    goto recv_again;
	}

      conn_info.byteorder = conn_info.buf->buf[0];
      setup_result = dpscap_ConnSetup(&conn_info, conn_info.buf->buf, len);
      if (setup_result <= 0)
	{
	  proc_agent_finalize(-1);
	  dprintf("Wrong byte order\n");
	  RETURN(-1);
	}
      gs_main_run_string_with_length(gs_main_instance_default(),
				     run_string,
				     strlen(run_string),
				     gs_main_instance_default()->user_errors,
				     &exit_code,
				     &error_object);
      proc_agent_finalize(-1);
      RETURN(-1);
    }
  else 
    {
      /** Parent */
      server_agent_pids = g_set_add(server_agent_pids, (gpointer)pid);
      return 0;
    }
}
#undef RETURN

private void
proc_server_wait_agent(int sig)
{
  pid_t pid;
  int status;

  while ((pid = waitpid(-1, &status, WNOHANG)) > 0)
    ;				/* Do NOTHING */
  server_agent_pids = g_set_remove(server_agent_pids, (gpointer)pid);
  signal(SIGCHLD, proc_server_wait_agent); /* Re-set */
}

private void
proc_agent_finalize(int sig)
{
  if (DGS_DEBUG) 
    {
      const char * signame = util_signal_int2str(sig);
      fprintf(stderr, "proc_agent_finalize: %s\n", signame);
    }
  if (conn_info.buf)
    {
      buf_finalize(conn_info.buf);
    }
  g_hash_table_destroy(conn_info.cpsid_to_cxid_table);
  g_hash_table_destroy(conn_info.cxid_to_cpsid_table);
  g_hash_table_destroy(conn_info.cpsid_to_mask_table);
  g_hash_table_destroy(conn_info.cpsid_to_nextMask_table);
  /* TODO: Set should be destroyed */

  conn_insock_close(conn_info.sock);
  /* TODO: Send connection shutdown info to client */
  if (sig == 0)
    {
      gx_device_X * xdev = (gx_device_X *)gs_currentdevice(igs);
      while (xdev->regular_fonts) {
	x11fontmap *font = xdev->regular_fonts;

	xdev->regular_fonts = font->next;
	if (font->std_names)
	  XFreeFontNames(font->std_names);
	if (font->iso_names)
	  XFreeFontNames(font->iso_names);
	gs_free(font->x11_name, sizeof(char), strlen(font->x11_name) + 1,
		"x11_font_x11name");
	gs_free(font->ps_name, sizeof(char), strlen(font->ps_name) + 1,
		"x11_font_psname");

	gs_free((char *)font, sizeof(x11fontmap), 1, "x11_fontmap");
      }
      while (xdev->symbol_fonts) {
	x11fontmap *font = xdev->symbol_fonts;

	xdev->symbol_fonts = font->next;
	if (font->std_names)
	  XFreeFontNames(font->std_names);
	if (font->iso_names)
	  XFreeFontNames(font->iso_names);
	gs_free(font->x11_name, sizeof(char), strlen(font->x11_name) + 1,
		"x11_font_x11name");
	gs_free(font->ps_name, sizeof(char), strlen(font->ps_name) + 1,
		"x11_font_psname");

	gs_free((char *)font, sizeof(x11fontmap), 1, "x11_fontmap");
      }
      while (xdev->dingbat_fonts) {
	x11fontmap *font = xdev->dingbat_fonts;

	xdev->dingbat_fonts = font->next;
	if (font->std_names)
	  XFreeFontNames(font->std_names);
	if (font->iso_names)
	  XFreeFontNames(font->iso_names);
	gs_free(font->x11_name, sizeof(char), strlen(font->x11_name) + 1,
		"x11_font_x11name");
	gs_free(font->ps_name, sizeof(char), strlen(font->ps_name) + 1,
		"x11_font_psname");

	gs_free((char *)font, sizeof(x11fontmap), 1, "x11_fontmap");
 
      }
      XDestroyWindow(shared_dpy, conn_info.agent_win);
      XCloseDisplay(shared_dpy);
    }
  gs_exit(0);
}

private void
proc_server_finalize(int sig)
{
  if (DGS_DEBUG) 
    {
      const char * signame = util_signal_int2str(sig);
      fprintf(stderr, "proc_server_finalize: %s...\n", signame);
    }
  conn_atom_cleanup();
  XDestroyWindow(shared_dpy, server_win_tcp);
  XDestroyWindow(shared_dpy, server_win_unix);
  conn_sock_finalize(&server_sockaddr_unix, UNIXCONN);
  conn_sock_finalize(&server_sockaddr_tcp, TCPCONN);
  conn_server_close(server_socket_tcp);
  conn_server_close(server_socket_unix);
  XCloseDisplay(shared_dpy);
  if (DGS_DEBUG) 
    fprintf(stderr, "done\n");
  gs_exit(-1);
}

private void
proc_server_setup_signal(void)
{
  signal(SIGCHLD, proc_server_wait_agent);
  signal(SIGHUP,  proc_server_finalize);
  signal(SIGINT,  proc_server_finalize);
  signal(SIGTERM, proc_server_finalize);
  signal(SIGSEGV, proc_server_finalize);
  signal(SIGUSR1, proc_server_control);
}

private void
proc_agent_setup_signal(void)
{
  signal(SIGHUP,  proc_agent_finalize);
  signal(SIGINT,  proc_agent_finalize);
  signal(SIGTERM, proc_agent_finalize);
  signal(SIGSEGV, proc_agent_finalize);
  signal(SIGUSR1, proc_agent_control);
}

/* print agent pids */
private void proc_server_control_pids(void);
private void
proc_server_control(int sig)
{
  proc_server_control_pids();
}

private void proc_agent_control_ppid_print(gpointer data, gpointer user_data);
private void
proc_server_control_pids(void)
{
  if (0 == g_set_count(server_agent_pids))
    {
      dprintf("agent pids: no agents\n");
    }
  else
    {
      dprintf("agent pids: \n");
      g_set_foreach(server_agent_pids, 
		    proc_agent_control_ppid_print, NULL);
    }
}

private void
proc_agent_control_ppid_print(gpointer data, gpointer user_data)
{
  dprintf1("%d\n", (int)data);
}


private void proc_agent_control_ppid(void);
private void
proc_agent_control(int sig)
{
  proc_agent_control_ppid();
}

private void 
proc_agent_control_ppid(void)
{
  dprintf1("server pid: %d\n", getppid());
}

/* I wonder why this function is needed. */
private Window 
proc_agent_setup_window(Display ** dpy, int agent_number)
{
  int i;
  Window tmp_win;

  /* Don't close but reopen...why this works? */
  *dpy = XOpenDisplay(NULL);	
  if (!*dpy) {
    fprintf(stderr, "Cannot open X Display(agent): %s\nABORTING!\n", 
            XDisplayName(NULL));
    return -1;
    }

  for (i = 0; i <= agent_number; i++)
    {
      tmp_win = XCreateSimpleWindow(*dpy, DefaultRootWindow(*dpy),
       			 0, 0, 1, 1, /* min X window size */
       			 0, /* no border either */
       			 BlackPixel(shared_dpy, DefaultScreen(*dpy)),
       			 WhitePixel(shared_dpy, DefaultScreen(*dpy)));
      if (!tmp_win) {
        fprintf(stderr, "Unable to create agent comm window.\n");
        XCloseDisplay(*dpy);
        return -1;
      }
    }
  if (!tmp_win) {
    fprintf(stderr, "Unable to create agent comm window.\n");
    XCloseDisplay(*dpy);
    return -1;
  }
  XSelectInput(*dpy, tmp_win, NoEventMask);
  return tmp_win;

}

/* --- dpscap_* --- */

/* --- dpscap_Conn --- */
private void dpscap_ConnSetup_display(xCAPConnSetupReq *req);
private void dpscap_ConnSetup_reply_success(xCAPConnSuccess * r, Window win) ;
private void dpscap_ConnSetup_reply_failed(xCAPConnFailed * r, int reason_length);

private int
dpscap_ConnSetup(gsdpsx_conn_info *conn, 
       	  unsigned char *msg, int len)
{
    xCAPConnSetupReq *req = (xCAPConnSetupReq*)msg;
    int used;
    xCAPConnSuccess reply_success;
    xCAPConnFailed reply_failed;
    const char * reason;

    /* Intel: 'l', PowerPC and Sparc: 'B' */
    if ((conn->byteorder != 'l' && conn->byteorder != 'B')) 
      {
	reason = "Wrong byte oroder";
	dpscap_ConnSetup_reply_failed(&reply_failed, strlen(reason));
	send(conn->sock, (const void *)&reply_failed, sizeof(xCAPConnFailed), 0);
	send(conn->sock, (const void *)reason, strlen(reason), 0);
	return -1;
      }

    used = sz_xCAPConnSetupReq;
    if (len-used < 0)
      {
	reason = "Request is too short";
	dpscap_ConnSetup_reply_failed(&reply_failed, strlen(reason));
	send(conn->sock, (const void *)&reply_failed, sizeof(xCAPConnFailed), 0);
	send(conn->sock, (const void *)reason, strlen(reason), 0);
        return -1;
      }

    /* Byteorder filtering */
    if (util_byteorder_check(conn))
      {
	req->flags = ntohs(req->flags);
	req->libraryversion = ntohl(req->libraryversion);
	req->authProtoNameLength = ntohs(req->authProtoNameLength);
	req->authProtoDataLength = ntohs(req->authProtoDataLength);
	req->displayStringLength = ntohs(req->displayStringLength);
	req->nodeStringLength = ntohs(req->nodeStringLength);
	req->transportStringLength = ntohs(req->transportStringLength);
	req->display = ntohs(req->display);
	req->screen = ntohs(req->screen);
	req->reserved = ntohs(req->reserved);
	req->clientWindow = ntohl(req->clientWindow);
      }
	  
    /* check that the display name is there */
    used += req->displayStringLength;
    if (len-used < 0 || req->displayStringLength > 255)
      {
	/* we can't deal with that */
	reason = "Wrong display name";
	dpscap_ConnSetup_reply_failed(&reply_failed, strlen(reason));
	send(conn->sock, (const void *)&reply_failed, sizeof(xCAPConnFailed), 0);
	send(conn->sock, (const void *)reason, strlen(reason), 0);
        return -1;
      }

    /* Set DISPLAY environment variable */
    dpscap_ConnSetup_display(req);
    conn->clientWindow = req->clientWindow;

    /* alright, time to send a reply back */
    dpscap_ConnSetup_reply_success(&reply_success, conn->agent_win);
    send(conn->sock, (const void *)&reply_success, reply_success.reasonLength, 0);

    /* initialize outputs */
    gsdpsx_std_init();


    return used;
}

private void
dpscap_ConnSetup_display(xCAPConnSetupReq *req)
{
  unsigned char * msg = (unsigned char * )req;
  int off;
  char displayname[256];
  /* The display name might not be there and it certainly won't be NULL
     terminated.  We don't need to worry about overwriting what is after it */
  if (req->displayStringLength) 
    {
      off = sz_xCAPConnSetupReq 
        + req->authProtoNameLength 
        + req->authProtoDataLength;
      memcpy(displayname, msg+off, req->displayStringLength);
      displayname[req->displayStringLength] = '\0';
      if (DGS_DEBUG) 
        dprintf1("setting DISPLAY to '%s'\n", displayname);
      setenv("DISPLAY", displayname, 1);
    }
}

private void
dpscap_ConnSetup_reply_failed(xCAPConnFailed * r, int reason_length)
{
  int swap = util_byteorder_check(&conn_info);
  r->success = FALSE;
  r->reasonLength = reason_length;
  if (swap)
    {
      r->additionalLength = htons(0);
      r->serverVersion = htonl(DPSPROTOCOLVERSION);
      r->dpscapVersion = DPSCAPPROTOVERSION;
      r->pad = 0;
      r->reserved = htons(0);
    }
  else
    {
      r->additionalLength = 0;
      r->serverVersion = DPSPROTOCOLVERSION;
      r->dpscapVersion = DPSCAPPROTOVERSION;
      r->pad = 0;
      r->reserved = 0;
    }
}

private void
dpscap_ConnSetup_reply_success(xCAPConnSuccess * r, Window win)
{
  int swap = util_byteorder_check(&conn_info);

  r->success = TRUE;
  r->reasonLength = sizeof(xCAPConnSuccess);
  /* This field is ignored by the client. */
  
  if (swap)
    {
      r->additionalLength = htons(0);
      r->serverVersion = htonl(DPSPROTOCOLVERSION);
      r->dpscapVersion = DPSCAPPROTOVERSION;
      r->reserved = 0;
      r->flagsUsed = htons(0);
      r->preferredNumberFormat = htonl(0); /*!*/
      r->floatingNameLength = htonl(0); /*! whats this? */
      r->agentWindow = htonl(win); /* only used to resume suspended contexts */
    }
  else
    {
      r->additionalLength = 0;
      r->serverVersion = DPSPROTOCOLVERSION;
      r->dpscapVersion = DPSCAPPROTOVERSION;
      r->reserved = 0;
      r->flagsUsed = 0;
      r->preferredNumberFormat = 0; /*!*/
      r->floatingNameLength = 0; /*! whats this? */
      r->agentWindow = win; /* only used to resume suspended contexts */
    }
}


/* --- dpscap_Notify --- */
private void dpscap_Notify_grab(gsdpsx_conn_info *conn, xCAPNotifyReq *req);
private void dpscap_Notify_ungrab(gsdpsx_conn_info *conn, xCAPNotifyReq *req);
private void dpscap_Notify_freegc(gsdpsx_conn_info *conn, xCAPNotifyReq *req);
private void dpscap_Notify_sync(gsdpsx_conn_info *conn, xCAPNotifyReq *req);
private void dpscap_Notify_pause(gsdpsx_conn_info *conn, xCAPNotifyReq *req,
				 enum scheduling_action * sched_act);

private int
dpscap_Notify(gsdpsx_conn_info *conn, unsigned char *msg, int remaining,
	      enum scheduling_action * sched_act)
{
    xCAPNotifyReq *req = (xCAPNotifyReq*)msg;
    int swap = util_byteorder_check(conn);
    short length = req->length;
    /* Byteorder filtering (stage 1)*/
    if (swap)
      length = ntohs(req->length);
    
    /* check for the whole message... must do first check to ensure
       that you can do the second */
    if (remaining < sz_xCAPNotifyReq || (length*4 > remaining))
      return 0;
    
    /* Byteorder filtering (stage 2)*/
    if (swap)
      {
	req->length = length;
	req->cxid = ntohl(req->cxid);
	req->notification = ntohl(req->notification);
	req->data = ntohl(req->data);
	req->extra = ntohl(req->extra);
      }

  switch (req->notification) 
    {
    case DPSCAPNOTE_GRAB:
      dpscap_Notify_grab(conn, req);
      break;
    case DPSCAPNOTE_UNGRAB:
      dpscap_Notify_ungrab(conn, req);
      break;
    case DPSCAPNOTE_FREEGC:
      dpscap_Notify_freegc(conn, req);
      break;
    case DPSCAPNOTE_SYNC:
      dpscap_Notify_sync(conn, req);
      break;
    case DPSCAPNOTE_PAUSE:
      dpscap_Notify_pause(conn, req, sched_act);
      break;
    default:
      dprintf1("ignoring unknown xCAPNotify %X\n", (unsigned)req->notification);
    }
  return (req->length)*4;
}

private void
dpscap_Notify_grab(gsdpsx_conn_info *conn, xCAPNotifyReq *req)
{
  /* TODO */
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] Notify->grab is not implemented\n");
}

private void
dpscap_Notify_ungrab(gsdpsx_conn_info *conn, xCAPNotifyReq *req)
{
  /* TODO */
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] Notify->ungrab is not implemented\n");
}
private void
dpscap_Notify_freegc(gsdpsx_conn_info *conn, xCAPNotifyReq *req)
{
  /* TODO */
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] Notify->freegc is not implemented\n");
}

private void
dpscap_Notify_sync(gsdpsx_conn_info *conn, xCAPNotifyReq *req)
{

  XEvent event;
  gx_device_X * xdev = (gx_device_X *)gs_currentdevice(igs);
  if (!xdev)
    fprintf(stderr, "Cannot find device in dpscap_Notify_sync\n");
  event.xclient.type = ClientMessage;
  event.xclient.serial = req->data;
  event.xclient.send_event = True;
  event.xclient.message_type = XInternAtom(xdev->dpy, DPSCAP_TYPE_SYNC,False);
  event.xclient.format = 8;

  if (util_byteorder_check(conn))
    event.xclient.data.l[0] = htonl(req->data);
  else
    event.xclient.data.l[0] = req->data;

  event.xclient.display = xdev->dpy;
  event.xclient.window = conn->clientWindow;
  XSendEvent(event.xclient.display, event.xclient.window, False, 0, &event);
  XSync(event.xclient.display, False);
}

private void dpscap_Notify_pause_check_callback_set(gpointer data, gpointer user_data);
private void dpscap_Notify_pause_remove(GSet ** base_set, GSet ** remove_set);
private Bool dpscap_Notify_pause_check_callback_x(Display *xdpy, XEvent *event, XPointer arg);
private void
dpscap_Notify_pause(gsdpsx_conn_info *conn, xCAPNotifyReq *req,
		    enum scheduling_action * sched_act)
{
  /* a PAUSE requires that we wait until we receive an X event
     (on the X comm link) to resume.  This ensures that X has caught up
     with the window creation or other changes */
  /* XEvent event; */
  conn->pause_cxids_set = g_set_add(conn->pause_cxids_set, (gpointer)req->cxid);
  /*! yielding to other threads here would be nice, 
    but this will require some fancy work */
  /*! this could eat the event from another context... 
    we need to be careful of that,
    either by creating a different window per connection, or 
    waking the other up */
  /*! we might actually be okay, it turns out */
  /* XIfEvent(shared_dpy, &event, 
	   dpscap_Notify_pause_check_callback_x,
	   (XPointer)req->cxid); */
  if (!dpscap_Notify_pause_check(&conn_info, req->cxid, 0))
    {
      int next = 0;
      int steady = 1;
      int cpsid = (int)g_hash_table_lookup(conn->cxid_to_cpsid_table,
					   (gpointer)req->cxid); 
      if (!status_send_if_masked(conn, cpsid, PSFROZEN, next))
	status_send_if_masked(conn, cpsid, PSFROZEN, steady);
      *sched_act = SCHED_ACT_YIELD;
    }
}

private Bool
dpscap_Notify_pause_check(gsdpsx_conn_info *conn, int cxid, int sync)
{
  GSet * remove_set = NULL;
  XEvent event;
  Bool resutl;

  if (!(conn->pause_cxids_set))
    return True;
  if (g_set_count(conn->pause_cxids_set) == 0)
    return True;
  
  if (cxid)
    {
      if (sync)
	{
	  XIfEvent(shared_dpy, 
		   &event,
		   dpscap_Notify_pause_check_callback_x,
		   (XPointer)cxid);
	  conn->pause_cxids_set = g_set_remove(conn->pause_cxids_set,
					       (gpointer)cxid);
	  return True;
	}
      else
	{
	  resutl = XCheckIfEvent(shared_dpy, 
			       &event,
			       dpscap_Notify_pause_check_callback_x,
			       (XPointer)cxid);
	  if (resutl)
	    conn->pause_cxids_set = g_set_remove(conn->pause_cxids_set,
						 (gpointer)cxid);
	  return resutl;
	}
    }
  else 
    {
      if (sync)
	  return False;		/* What should I do? */
      else
	{
	  g_set_foreach(conn->pause_cxids_set,
			dpscap_Notify_pause_check_callback_set,
			&remove_set);
	  if (g_set_count(remove_set) ==  0)
	    return False;
	  dpscap_Notify_pause_remove(&(conn->pause_cxids_set), &remove_set);
	  return True;
	}

    }
}
private void
dpscap_Notify_pause_check_callback_set(gpointer data, gpointer user_data)
{
  XEvent event;
  CONTEXTXID cxid = (CONTEXTXID)data;
  Bool result = XCheckIfEvent(shared_dpy, 
			      &event,
			      dpscap_Notify_pause_check_callback_x,
			      (XPointer)cxid);
  if (result)
    *((GSet **)user_data) = g_set_add(*((GSet **)user_data) , 
				      (gpointer)cxid);
  
}
/* checks to see if the event is a resume on the cxid 
 Was: WaitForResumeProc */
private Bool
dpscap_Notify_pause_check_callback_x(Display *xdpy, XEvent *event, XPointer arg)
{

  if (event->xany.window == conn_info.agent_win 
      && event->type == ClientMessage
      && event->xclient.message_type == conn_info.resume_atom 
      && event->xclient.format == 32 
      && event->xclient.data.l[0] == (CONTEXTXID)arg)
    /* the commented out is context X id... we don't support 
       multiple contexts! */
    {
      /*dprintf("got RESUME Message\n");*/
      return True;
    }
  else
    return False;
}
private void
dpscap_Notify_pause_remove(GSet ** base_set, GSet ** remove_set)
{
  GSList * slist = (GSList *)*remove_set;
  guint length = g_slist_length(slist);
  int i;
  CONTEXTXID cxid;
  for (i = 0; i < length; i++)
    {
      cxid = (CONTEXTXID)g_slist_nth_data(slist, i);
      *base_set = g_set_remove(*base_set, (gpointer)cxid);
    }
  for (i = 0; i < length; i++)
    {
      cxid = (CONTEXTXID)g_slist_nth_data(slist, 0);
      slist = g_slist_remove(slist, (gpointer)cxid);
    }
  *remove_set = (GSet *)slist;
}

/* --- dpscap_SetArg --- */
private int
dpscap_SetArg(gsdpsx_conn_info *conn, unsigned char *pmsg, int remaining)
{
  /* Still ... TODO */
  xCAPSetArgReq * req = (xCAPSetArgReq *)pmsg;
  short length = req->length;
  int result;
  int swap = util_byteorder_check(conn);
  if (swap)
    length = ntohs(req->length);
  
  result= length * 4;
  if (remaining < sz_xCAPSetArgReq || (result > remaining))
    return 0;
  
  if (swap)
    {
      req->length = length;
      req->arg = ntohl(req->arg);
      req->val = ntohl(req->val);
    }
  switch (req->arg)
    {
    case AGENT_ARG_SMALLFONTS:
      switch (req->val)
	{
	case AGENT_SMALLFONTS_ACCURATE:
	case AGENT_SMALLFONTS_FAST:
	  break;
	default:
	  fprintf(stderr,
		  "Wrong value for AGENT_ARG_SMALLFONTS in dpscap_SetArg\n");
	}
      break;
    case AGENT_ARG_PIXMEM:
      switch (req->val)
	{
	case AGENT_PIXMEM_LIMITED:
	case AGENT_PIXMEM_MODERATE:
	case AGENT_PIXMEM_UNLIMITED:
	  break;
	default:
	  fprintf(stderr,
		  "Wrong value for AGENT_ARG_PIXMEM in dpscap_SetArg\n");
	}
      break;
    default:
      fprintf(stderr, "Wrong argument specification in dpscap_SetArg\n");
    }
    return result;
}

/* --- dpscap function stubs (not implemented) --- */
private int
dpscap_FlushAgent(gsdpsx_conn_info *conn, unsigned char *pmsg, int remaining)
{
  /* TODO */
  xCAPFlushAgentReq * req = (xCAPFlushAgentReq * )pmsg;
  short length = req->length;
  int result;

  if (util_byteorder_check(conn))
    length = ntohs(req->length);

  result = length*4;
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] FlushAgent is not implemented\n");
  if (remaining < sz_xCAPFlushAgentReq
      || (result > remaining))
    return 0;
  else
    return result;
}


/* --- dps_CreateContext --- */
private int
dps_CreateContext(gsdpsx_conn_info *conn,
		  unsigned char *msg, int remaining,
		  enum scheduling_action * sched_act)
{
    xPSCreateContextReq *req = (xPSCreateContextReq*)msg;
    int swap = util_byteorder_check(conn);
    short length = req->length;
    /* Byteorder filtering (stage 1)*/
    if (swap)
      length = ntohs(req->length);
    /* check for the whole message... must do first check to ensure
       that you can do the second */
    if (remaining < sz_xPSCreateContextReq || (length*4 > remaining))
        return 0;

    /* Byteorder filtering (stage 2)*/
    if (swap)
      {
	req->length = length;
	req->cxid = ntohl(req->cxid);
	req->sxid = ntohl(req->sxid); /* Not used yet */
	req->drawable = ntohl(req->drawable);
	req->gc = ntohl(req->gc); /* Not used yet */
	req->x = ntohs(req->x);
	req->y= ntohs(req->y);
	req->eventmask = ntohl(req->eventmask);	/* Not used yet */
	req->cmap = ntohl(req->cmap); /* Not used yet */
	req->redmax = ntohl(req->redmax);/* Not used yet */
	req->redmult = ntohl(req->redmult);/* Not used yet */
	req->greenmax = ntohl(req->greenmax);/* Not used yet */
	req->greenmult = ntohl(req->greenmult);/* Not used yet */
	req->bluemax = ntohl(req->bluemax);/* Not used yet */
	req->bluemult = ntohl(req->bluemult);/* Not used yet */
	req->colorbase = ntohl(req->colorbase);/* Not used yet */
	req->graymax = ntohl(req->graymax);/* Not used yet */
	req->graymult = ntohl(req->graymult);/* Not used yet */
	req->graybase = ntohl(req->graybase);/* Not used yet */
	req->actual = ntohl(req->actual);/* Not used yet */
      }
    
    /* cxid, XID context */
    conn->ctx_init_parameter.cxid = req->cxid, 

    /* set device drawable */
    conn->ctx_init_parameter.drawable = req->drawable;

    /* initial matrix, should it be needed */
    conn->ctx_init_parameter.offset_x = req->x;
    conn->ctx_init_parameter.offset_y = req->y;
    
    * sched_act = SCHED_ACT_FORK;
    return (req->length)*4;
}

private void
dps_CreateContext_reply(gsdpsx_conn_info *conn, int cpsid)
{
  xPSCreateContextReply r;
  int swap = util_byteorder_check(conn);

  r.type = X_Reply;
  if (swap)
    {
       r.sequenceNumber = htons(0); /* whoop.  not in a server */
       r.length = htonl((sizeof(r)-sz_xGenericReply)/4);
       r.cpsid = htonl(cpsid);
    }
  else
    {
      r.sequenceNumber = 0; /* whoop.  not in a server */
      r.length = (sizeof(r)-sz_xGenericReply)/4;
      r.cpsid = cpsid;
    }
  send(conn->sock, (void*)&r, sizeof(r), 0);
}


/* --- gsdpsx stream IO --- */
/* code to deal with DPS/X11 input streams! */

private int read_process_handle(gsdpsx_conn_info *conn, 
				unsigned char *pmsg, 
				int remaining,
				enum scheduling_action *);
private void read_process_const(stream_cursor_write *pw, 
				const int length, const byte * data);
private void read_process_dump_unhandled(int used, 
					 int len,
					 unsigned char *cur);

private void read_process_input_forward(enum scheduling_action sched_act,
					gsdpsx_conn_buf * buf,
					stream_cursor_write *pw);

private int read_process_try(gsdpsx_conn_info *conn);

private int
gsdpsx_std_read_process(stream_state *st, stream_cursor_read *ignore_pr,
			stream_cursor_write *pw, bool last)
{
  gsdpsx_conn_info * conn = &conn_info;
  gsdpsx_conn_buf * buf = conn->buf;
  enum scheduling_action sched_act = SCHED_ACT_NOTHING;
  gx_device_X * xdev;
  int len;

  int cpsid = gsdpsx_get_current_context_index();
  CONTEXTXID cxid = (CONTEXTXID)g_hash_table_lookup(conn->cpsid_to_cxid_table,
						    (void *)cpsid);
  int newest_cpsid = gsdpsx_get_newest_context_index();

  if (pw->limit - pw->ptr < fork_op_length) 
    {
      /* Buffer is too small */
      if (DGS_DEBUG)
	dprintf("Stream buffer is too small.\n");
      return 1;
    }
  if (cpsid == conn->context_state_index_primary
      && conn->context_state_index_last != conn->context_state_index_primary)
    {
      static int detach = 1;
      if (detach)
	{
	  sched_act = SCHED_ACT_DETACH;
	  detach = 0;
	}
      else
	sched_act = SCHED_ACT_YIELD;
      goto forward;
    }
  else if (cpsid != conn->context_state_index_last)
    {
      /* Context switing occured */
      /* why context switching happens?  */
      /* Reason: fork */
      dpscap_Notify_pause_check(&conn_info, 0, 0);
      if (conn->ctx_init_parameter.cxid != 0)
	{
	  /* Associate gs_interp_context_state_current<->cxid */
	  g_hash_table_insert(conn->cxid_to_cpsid_table,
			      (void *)(conn->ctx_init_parameter.cxid),
			      (void *)cpsid);
	  g_hash_table_insert(conn->cpsid_to_cxid_table,
			      (void *)cpsid,
			      (void *)(conn->ctx_init_parameter.cxid));
	  /* Status mask table */
	  g_hash_table_insert(conn->cpsid_to_mask_table,
			      (void *)cpsid,
			      (void *)0);
	  g_hash_table_insert(conn->cpsid_to_nextMask_table,
			      (void *)cpsid,
			      (void *)0);
	  /* Initialize context using ctx_init_parameter */
	  ctx_init_parameter_install(&(conn->ctx_init_parameter));
	  /* Answer to the client */
	  dps_CreateContext_reply(conn, (int)cpsid);
	  /* Destory parameter */
	  ctx_init_parameter_initialize(&(conn->ctx_init_parameter));
	  gsdpsx_clear_newest_context ();
	  gsdpsx_request_context_index(0);
	  conn->context_state_index_last = cpsid;
	}
    }
  else if (newest_cpsid && (conn->ctx_init_parameter.cxid != 0))
    {
      gsdpsx_request_context_index(newest_cpsid);
      sched_act = SCHED_ACT_YIELD;
      goto forward;
    }
  else if (g_set_member(conn->pause_cxids_set, (gpointer)cxid))
    {
      int sync = 0;
      if (gsdpsx_count_ready_context() == 1)
	sync = 1;
      if (!dpscap_Notify_pause_check(&conn_info, cxid, sync))
	{
	  sched_act = SCHED_ACT_YIELD;
	  goto forward;
	}
    } 
  else if (g_set_member(conn->kill_cpsids_set, (gpointer)cpsid))
    {
      sched_act = SCHED_ACT_YIELD;
      conn->kill_cpsids_set = g_set_remove(conn->kill_cpsids_set,
					   (gpointer)cpsid);
      goto forward;
    }
  /* flush out any drawing commands in the queue */
  xdev = (gx_device_X *)gs_currentdevice(igs);
  if (!xdev)
    fprintf(stderr, "Cannot find device in gsdpsx_std_read_process\n");
  if (xdev->dpy)
    XFlush(xdev->dpy);

  while (sched_act == SCHED_ACT_NOTHING
	 && !(buf->inputstart)) 
    {
      unsigned char *cur;
      int used = 0;

      if (buf->len && !buf->isfrag)
	{
	  /* Data is available in the buffer */
	  len = buf->len;
	  cur = buf->curpos;
	} 
      else 
	{
	  /* Data is NOT available in the buffer 
	     => Get data via socket */
	  if (!read_process_try(conn))
	    {
	      int next = 0;
	      int steady = 1;
	      if (!status_send_if_masked(conn, cpsid, PSNEEDSINPUT, next))
		status_send_if_masked(conn, cpsid, PSNEEDSINPUT, steady);
	    }

	  len = buf_recv(buf, conn);
	  if (-1 == len)
	    break;
	  if (0) 
	    dprintf1("actually got data block of %d bytes.\n", len);
	  len += buf->len;
	  cur = buf->buf;
	  if (!len)
	    break;
	}

      /* Consume the data.
	 + If the data that should be consumed in the gs interpreter,
	 break the loop. 
	 + If the data is exhausted, 
	 break the loop to get new data via socket. */
      while (sched_act == SCHED_ACT_NOTHING
	     && (!(buf->inputstart) && len))
	{
	  used = read_process_handle(conn, cur, len, &sched_act);

	  /* if !used, then we had a controlled underflow, we need more data
	     if used < 0, then an error */
	  if (used <= 0)
	    break;

	  cur += used;
	  len -= used;
	} 

      /* error? */
      if (used < 0) {
	dprintf1("error occured (used = %d).  Trying to continue.\n", 
		 (int)used);
	len = 0;
      }

      /* there was some leftover data, shuffle it around */
      if (len)
	{
	  if (!(buf->inputstart) && (sched_act == SCHED_ACT_NOTHING))
	      buf_pack(buf, cur, len);
	  else 
	    {
	      buf_move_cursor(buf, cur, len);
	      buf->isfrag = 0;
	      if (0) 
		dprintf1("Queing %d bytes of messages for next trip!\n", len);
	    }
	} 
      else /* len == 0 */
	buf_reset(buf);

      /* flush out any drawing commands in the queue */
      if (xdev->dpy)
	XFlush(xdev->dpy);
    } /* while data */

  if (sched_act == SCHED_ACT_NOTHING && (!(buf->inputstart))) {
    dprintf("returning EOFC\n");
    proc_agent_finalize(0);
    return EOFC;
  }

  /*  Copying socket buffer -> interpreter stream buffer */
 forward:
  if (sched_act == SCHED_ACT_NOTHING)
    {
      int next = 0;
      int steady = 1;
      if (!status_send_if_masked(conn, cpsid, PSRUNNING, next))
	status_send_if_masked(conn, cpsid, PSRUNNING, steady);
    }
  read_process_input_forward(sched_act, buf, pw);
  return 1;		/* See strimpl.h */
  /* return (count == tocopy) ? 1 : 0; */
}

private int
read_process_handle(gsdpsx_conn_info *conn, 
		    unsigned char *pmsg, 
		    int remaining,
		    enum scheduling_action * sched_act)
{
  int used, dealt_with = 1;

  if (pmsg[0] == DPSXOPCODEBASE) 
    {
      switch(pmsg[1]) 
	{
	case X_PSGiveInput:
	  used = dps_GiveInput(conn, pmsg, remaining, sched_act);
	  break;
	case X_PSCreateContext:
	  used = dps_CreateContext(conn, pmsg, remaining, sched_act);
	  break;
	case X_PSSetStatusMask: 
	  used = dps_SetStatusMask(conn, pmsg, remaining);
	  break;
	case X_PSNotifyContext:
	  used = dps_NotifyContext(conn, pmsg, remaining, sched_act);
	  break;
	case X_PSDestroySpace:
	  used = dps_DestroySpace(conn, pmsg, remaining);
	  break;
	case X_PSInit:
	  used = dps_Init(conn, pmsg, remaining);
	  break;
	case X_PSCreateSpace:
	  used = dps_CreateSpace(conn, pmsg, remaining);
	  break;
	case X_PSGetStatus:
	  used = dps_GetStatus(conn, pmsg, remaining);
	  break;
	case X_PSReset:
	  used = dps_Reset(conn, pmsg, remaining);
	  break;
	case X_PSCreateContextFromID:
	  used = dps_CreateContextFromID(conn, pmsg, remaining);
	  break;
	case X_PSXIDFromContext:
	  used = dps_XIDFromContext(conn, pmsg, remaining);
	  break;
	case X_PSContextFromXID:
	  used = dps_ContextFromXID(conn, pmsg, remaining);
	  break;
	case X_PSCreateSecureContext:
	  used = dps_CreateSecureContext(conn, pmsg, remaining);
	  break;
	case X_PSNotifyWhenReady:
	  used = dps_NotifyWhenReady(conn, pmsg, remaining);
	  break;
	default:
	  {
	    const char * name = util_x_ps_request_enum2str(pmsg[1]);
	    if (name == NULL)
	      dprintf1("received unknown DPSX message %X.\n",
		       (unsigned)pmsg[1]);
	    else
	      dprintf2("received unhandled DPSX message %X:%s.\n",
		       (unsigned)pmsg[1], name);
	    dealt_with = 0;
	  }
	}
    } 
  else if (pmsg[0] == DPSCAPOPCODEBASE) 
    {
      switch(pmsg[1]) {
      case X_CAPFlushAgent:
	used = dpscap_FlushAgent(conn, pmsg, remaining);
	break;
      case X_CAPNotify:
	used = dpscap_Notify(conn, pmsg, remaining, sched_act);
	break;
      case X_CAPSetArg:
	used = dpscap_SetArg(conn, pmsg, remaining);
	break;			
      default:
	{
	  const char * name = util_x_cap_request_enum2str(pmsg[1]);
	  if (name == NULL)
	    dprintf1("received unknown DPSCAP message %X.\n",
		     (unsigned)pmsg[1]);
	  else
	    dprintf2("received unhandled DPSCAP message %X:%s.\n",
		     (unsigned)pmsg[1], name);
	  dealt_with = 0;
	}
      }
    } 
  else if (pmsg[0] == X_ChangeGC)
    used = dpsx_ChangeGC(conn, pmsg, remaining);
  else 
    dealt_with = 0;
 
#if DUMP_UNHANDLED_MESSAGES
  if (!dealt_with) {
    used = (((xPSGiveInputReq*)pmsg)->length)*4; /* remove all data */
    /*! send error */
    dprintf1("got data of %d bytes!\n", remaining);
    read_process_dump_unhandled(used, remaining, pmsg);
  } /* if (!dealt_with) */
#endif /* DUMP_UNHANDLED_MESSAGES */
  
  if (used > 0 && remaining-used < 0) {
    dprintf("ERROR: buffer underflow! something went haywire!\n");
    dprintf1("ERROR: used: %d\n", used);
    dprintf1("ERROR: remaining: %d\n", remaining);
    used = remaining;
  }
  
  return used;
}

private void
read_process_const(stream_cursor_write *pw, const int length, const byte * data)
{
  byte * dist = (pw->ptr+1);
  memcpy(dist, data, sizeof(byte)*length);
  pw->ptr += length;  
}

private void
read_process_dump_unhandled(int used, int len, unsigned char *cur)
{
  int j,k;
  for (j=0; j < used;) {
    for (k = 0; k < 16; k++) {
      if ((j+k)>=len)
        dprintf("   ");
      else
        dprintf1("%02X ", cur[j+k]);

      if (k == 7)
        dprintf("- ");
      else if (k == 3 || k == 11)
        dprintf(" ");
    }
    dprintf("    ");
    for (k = 0; k < 16; k++) {
      unsigned char c = cur[j+k];
      if ((j+k)<len)
        dprintf1("%c", (c >= 32 && c < 128) ? c : '.');

      if (k == 7) 
        dprintf("-");
    }
    dprintf("\n");
    j += 16;
  }
}

private void
read_process_input_forward(enum scheduling_action sched_act,
			       gsdpsx_conn_buf * buf,
			       stream_cursor_write *pw)
{
  switch (sched_act)
    {
    case SCHED_ACT_NOTHING:
      buf_input_forward(buf, pw);	
      break;
    case SCHED_ACT_FORK:
      read_process_const(pw, fork_op_length, fork_op_data);  
      break;
    case SCHED_ACT_YIELD:
      read_process_const(pw, yield_op_length, yield_op_data);  
      break;
    case SCHED_ACT_DETACH:
      read_process_const(pw, detach_op_length, detach_op_data);  
      break;
    default:
      dprintf("Wrong sched_act...abort\n");
      dprintf("Function: read_process_input_forward\n");
      read_process_const(pw, yield_op_length, yield_op_data);  
    }
}

private int
read_process_try(gsdpsx_conn_info *conn)
{
  fd_set rfds;
  struct timeval tv;
  FD_ZERO(&rfds);
  FD_SET(conn->sock, &rfds);
  tv.tv_sec = 0;
  tv.tv_usec = 0;
  return select(conn->sock + 1, &rfds, NULL, NULL, &tv);
}

/* code to deal with DPS/X11 result streams! */
private int
gsdpsx_std_write_process(stream_state *st, stream_cursor_read *pr,
			 stream_cursor_write *ignore_pw, bool last)
{
  uint count = pr->limit - pr->ptr;
  stream *s = (stream*)st;
  gsdpsx_conn_info * conn = (gsdpsx_conn_info *)(s->file);
  static Atom outputAtom = 0;
  gx_device_X * xdev = (gx_device_X *)gs_currentdevice(igs);

  if (!xdev || !xdev->dpy)
    return -1;

  if (!outputAtom)
    outputAtom = XInternAtom(xdev->dpy, DPSCAP_TYPE_PSOUTPUT_LEN, False);

  while ( count != 0 )
    {
      int towrite = min(count, DPSCAP_DATA_LEN);
      XEvent event;
      DPSCAPOutputEvent *out;

      event.xclient.type = ClientMessage;
      event.xclient.serial = 0;
      event.xclient.send_event = True;
      event.xclient.message_type = outputAtom;
      event.xclient.format = 8;
      out = (DPSCAPOutputEvent*)event.xclient.data.l;
      {
	int cpsid = gsdpsx_get_current_context_index();
	CONTEXTXID cxid = (CONTEXTXID)g_hash_table_lookup(conn->cpsid_to_cxid_table,
							  (void *)cpsid);
	if (util_byteorder_check(conn))
	  out->cxid = htonl(cxid);
	else
	  out->cxid = cxid;
      }
      memcpy(out->data, pr->ptr+1, towrite);
      out->data[DPSCAP_DATA_LEN] = towrite;
      event.xclient.display = xdev->dpy;
      event.xclient.window = conn->clientWindow;
      if (0) {
        char data[21];
        memcpy(data,pr->ptr+1,towrite);
        data[towrite] = '\0';
        dprintf2("sent %d bytes of results\n  ->%s<-\n", towrite, data);
      }

      XSendEvent(event.xclient.display, event.xclient.window, False, 0, &event);
      pr->ptr += towrite;
      count -= towrite;
    }
  XSync(xdev->dpy, False);
  return 0;
}

extern int iodev_stdin_open(P4(gx_io_device *, const char *, stream **,
       			gs_memory_t *));
private int
gsdpsx_stdin_open(gx_io_device *iodev, const char *access, stream **ps,
       	   gs_memory_t *mem)
{
  int code = iodev_stdin_open(iodev, access, ps, mem);
  stream *s = *ps;
  if ( code < 0 )
    return code;
  /* #error FLUSH THE QUEUE, maybe that will help
     #error also check that we need all the below */
  s->procs.process = gsdpsx_std_read_process;
  s->procs.seek = s_std_noseek;
  s->procs.available = s_std_noavailable;
  s->procs.switch_mode = s_std_switch_mode;
  s->procs.close = s_std_close;
  s->procs.flush = s_std_read_flush;
  s->file = (FILE*)(&conn_info); /***** GET RIGHT INFO! *****/
  return 0;
}

extern int iodev_stderr_open(P4(gx_io_device *, const char *, stream **,
       			 gs_memory_t *));
private int
gsdpsx_stderr_open(gx_io_device *iodev, const char *access, stream **ps,
       	    gs_memory_t *mem)
{	
  int code = iodev_stderr_open(iodev, access, ps, mem);
  stream *s = *ps;
  if ( code < 0 )
    return code;
  s->procs.process = gsdpsx_std_write_process;
  s->procs.seek = s_std_noseek;
  s->procs.available = s_std_noavailable;
  s->procs.switch_mode = s_std_switch_mode;
  s->procs.close = s_std_close;
  s->procs.flush = s_std_write_flush;
  s->file = (FILE*)(&conn_info); /***** GET RIGHT INFO! *****/
  return 0;
}

extern int iodev_stdout_open(P4(gx_io_device *, const char *, stream **,
       			 gs_memory_t *));
private int
gsdpsx_stdout_open(gx_io_device *iodev, const char *access, stream **ps,
       	    gs_memory_t *mem)
{	int code = iodev_stdout_open(iodev, access, ps, mem);
 stream *s = *ps;
 if ( code < 0 )
   return code;
 s->procs.process = gsdpsx_std_write_process;
 s->procs.seek = s_std_noseek;
 s->procs.available = s_std_noavailable;
 s->procs.switch_mode = s_std_switch_mode;
 s->procs.close = s_std_close;
 s->procs.flush = s_std_write_flush;
 s->file = (FILE*)(&conn_info); /***** GET RIGHT INFO! *****/
 return 0;
}

/* Patch stdout/err to use our stream. */
private void
gsdpsx_std_init(void)
{
  gs_findiodevice((const byte *)"%stdout", 7)->procs.open_device =
    gsdpsx_stdout_open;

  gs_findiodevice((const byte *)"%stderr", 7)->procs.open_device =
    gsdpsx_stderr_open;

  gs_findiodevice((const byte *)"%stdin", 6)->procs.open_device =
    gsdpsx_stdin_open;
}

/* --- dps_GiveInput --- */
private int
dps_GiveInput    (gsdpsx_conn_info *conn,
		  unsigned char *pmsg, int remaining,
		  enum scheduling_action * sched_act)
{
  xPSGiveInputReq *req = (xPSGiveInputReq*)pmsg;
  int used;
  short length = req->length;
  int cpsid = gsdpsx_get_current_context_index();
  int new_cpsid;
  CONTEXTXID cxid = 
    (CONTEXTXID)g_hash_table_lookup(conn->cpsid_to_cxid_table,
				    (void *)cpsid);
  int swap = util_byteorder_check(conn);
  int req_cxid_backup;
  int req_cxid_swapped;

  /* Byteorder filtering (stage 1)*/
  if (swap)
    length = ntohs(req->length);

  used = length*4;

  if (remaining < used) 
    {
      /* get the rest of the data */
      if (0) 
	dprintf2("incomplete message, remaining is %d, needed %d\n",
		 (int)remaining, 
		 (int)used) ;
      return 0;
    }
  
  /* Byteorder filtering (stage 1.5)*/
  if (swap)
    {
      req_cxid_backup = req->cxid;
      req_cxid_swapped = ntohl(req->cxid);
    }
  else
    {
      req_cxid_backup = req->cxid;
      req_cxid_swapped = req->cxid;
    }
  
  req->cxid = req_cxid_swapped;
  if (req->cxid != cxid)
    {
      new_cpsid = (int)g_hash_table_lookup(conn->cxid_to_cpsid_table,
					   (void *)req->cxid);
      *sched_act = SCHED_ACT_YIELD;
      gsdpsx_request_context_index(new_cpsid);
      req->cxid = req_cxid_backup;
      return 0;
    }
  req->cxid = req_cxid_backup;
  
  if (swap)
    {
      req->length = length;
      req->nunits = ntohs(req->nunits);
      req->pad = ntohs(req->pad);
      req->cxid = req_cxid_swapped;
    }

  buf_input_set(conn->buf, (char*)(req+1), req->nunits);
  return used;
}

private void dps_XIDFromContext_reply(gsdpsx_conn_info *conn, CONTEXTXID cxid);

private int
dps_XIDFromContext(gsdpsx_conn_info *conn,
       	    unsigned char *pmsg, int remaining)
{
  xPSXIDFromContextReq *req = (xPSXIDFromContextReq *)pmsg;
  CONTEXTXID cxid;
  int swap = util_byteorder_check(conn);
  int length = req->length;
  int result;
  if (swap)
    length = ntohs(req->length);
  
  result = length*4;
  if (remaining < sz_xPSXIDFromContextReq
      || (result > remaining))
    return 0;
  
  if (swap)
    {
      req->length = length;
      req->cpsid = ntohl(req->cpsid);
    }
  cxid = (CONTEXTXID)g_hash_table_lookup(conn->cpsid_to_cxid_table,
				  (void *)req->cpsid);
  dps_XIDFromContext_reply(conn, cxid);
  return result;
}

private void 
dps_XIDFromContext_reply(gsdpsx_conn_info *conn, CONTEXTXID cxid)
{
  xPSXIDFromContextReply r;  
  int swap = util_byteorder_check(conn);
  r.type = X_Reply;
  if (swap)
    {
      r.sequenceNumber = htons(0);	/* TODO */
      r.length = htonl((sizeof(r)-sz_xGenericReply)/4);  
      r.cxid = htonl(cxid);
      r.sxid = htonl(0);		/* TODO */
    }
  else
    {
      r.sequenceNumber = 0;	/* TODO */
      r.length = (sizeof(r)-sz_xGenericReply)/4;  
      r.cxid = cxid;
      r.sxid = 0;		/* TODO */
    }
  send(conn->sock, (void*)&r, sizeof(r), 0);
}

private void dps_ContextFromXID_reply(gsdpsx_conn_info *conn, int cpsid);
private int
dps_ContextFromXID(gsdpsx_conn_info *conn,
       	    unsigned char *pmsg, int remaining)
{
  xPSContextFromXIDReq *req = (xPSContextFromXIDReq *)pmsg;
  int cpsid;
  int swap = util_byteorder_check(conn);
  int length = req->length;
  int result;

  if (swap)
    length = ntohs(req->length);

  result = length*4;
  if (remaining < sz_xPSContextFromXIDReq
      || (result > remaining))
    return 0;
  
  if (swap)
    {
      req->length = length;
      req->cxid = ntohl(req->cxid);
    }
  
  cpsid = (int)g_hash_table_lookup(conn->cxid_to_cpsid_table,
				   (void *)req->cxid);
  dps_ContextFromXID_reply(conn, cpsid);
  return result;
}
private void
dps_ContextFromXID_reply(gsdpsx_conn_info *conn, int cpsid)
{
  xPSContextFromXIDReply r;
  int swap = util_byteorder_check(conn);

  r.type = X_Reply;
  if (swap)
    {
      r.sequenceNumber = htons(0); /* TODO */
      r.length = htonl((sizeof(r)-sz_xGenericReply)/4);  
      r.cpsid = htonl((CARD32)cpsid);
    }
  else
    {
      r.sequenceNumber = 0;	/* TODO */
      r.length = (sizeof(r)-sz_xGenericReply)/4;  
      r.cpsid = (CARD32)cpsid;
    }
  send(conn->sock, (void*)&r, sizeof(r), 0);
}


/* --- dps_CreateContextFromID --- */
private void
dps_CreateContextFromXID_reply(gsdpsx_conn_info *conn);

private int
dps_CreateContextFromID(gsdpsx_conn_info *conn,
       		 unsigned char *pmsg, int remaining)
{
  xPSCreateContextFromIDReq *req = (xPSCreateContextFromIDReq *)pmsg;
  short length = req->length;
  int result;
  int swap = util_byteorder_check(conn);
  if (swap)
    length = ntohs(req->length);
  
  result = length * 4;
  if (remaining < sz_xPSCreateContextFromIDReq
      || (result > remaining))
    return 0;
  
  if (swap)
    {
      req->length = length;
      req->cpsid = ntohl(req->cpsid);
      req->cxid = ntohl(req->cxid);
    }
  
  g_hash_table_insert(conn->cxid_to_cpsid_table,
		      (void *)(req->cxid),
		      (void *)(req->cpsid));
  g_hash_table_insert(conn->cpsid_to_cxid_table,
		      (void *)(req->cpsid),
		      (void *)(req->cxid));
  
  dps_CreateContextFromXID_reply(conn);
  return result;
}

private void
dps_CreateContextFromXID_reply(gsdpsx_conn_info *conn)
{
  xPSCreateContextFromIDReply r;
  int swap = util_byteorder_check(conn);
  
  r.type = X_Reply;
  if (swap)
    {
      r.sequenceNumber = htons(0); /* TODO */
      r.length = htonl((sizeof(r)-sz_xGenericReply)/4);  
      r.sxid = htonl(0);	/* TODO */
    }
  else
    {
      r.sequenceNumber = 0; /* TODO */
      r.length = (sizeof(r)-sz_xGenericReply)/4;  
      r.sxid = 0;
    }
  send(conn->sock, (void*)&r, sizeof(r), 0);
}


/* --- dps_GetStatus --- */
private void dps_GetStatus_reply(gsdpsx_conn_info *conn,
				 CONTEXTXID cxid);
private int
dps_GetStatus    (gsdpsx_conn_info *conn,
       	   unsigned char *pmsg, int remaining)
{
  xPSGetStatusReq *req = (xPSGetStatusReq *)pmsg;
  short length = req->length;
  int result;
  int swap = util_byteorder_check(conn);
  if (swap);
    length = ntohs(req->length);
  
  result = length * 4;
  if (remaining < sz_xPSGetStatusReq || (result > remaining))
    return 0;
  
  if (swap)
    {
      req->length = length;
      req->cxid = ntohl(req->cxid);
    }
  dps_GetStatus_reply(conn, req->cxid);
  return result;
}

private void
dps_GetStatus_reply(gsdpsx_conn_info *conn, CONTEXTXID cxid)
{
  xPSGetStatusReply r;
  int swap = util_byteorder_check(conn);
  int current_cpsid = gsdpsx_get_current_context_index();
  int cpsid = (CONTEXTXID)g_hash_table_lookup(conn->cxid_to_cpsid_table,
					      (void *)cxid);
  r.type = X_Reply;

  if (swap)
    {
      r.sequenceNumber = htons(0);
      r.length = htonl((sizeof(r)-sz_xGenericReply)/4);  
    }
  else
    {
      r.sequenceNumber = htons(0);
      r.length = (sizeof(r)-sz_xGenericReply)/4;  
    }
  /* TODO:
     PSSTATUSERROR PSRUNNING PSNEEDSINPUT PSZOMBIE PSFROZEN */
  if (g_set_member(conn->pause_cxids_set, (gpointer)cxid))
    r.status = PSFROZEN;
  else if (cpsid == current_cpsid)
    r.status = PSNEEDSINPUT;
  else if (g_set_member(conn->kill_cpsids_set, (gpointer)cpsid))
    r.status = PSZOMBIE;
  else
    r.status = PSRUNNING;
  send(conn->sock, (void*)&r, sizeof(r), 0);
}

/* --- dps_SetStatusMask --- */
private void dps_SetStatusMask_dump(const char * header,
				   int mask);
private int 
dps_SetStatusMask(gsdpsx_conn_info *conn, unsigned char *pmsg, int remaining)
{
  /* TODO */
  xPSSetStatusMaskReq *req = (xPSSetStatusMaskReq*)pmsg;
  short length = req->length;
  int result;
  int swap = util_byteorder_check(conn);
  int cpsid;
  int mask;

  if (swap)
    length = ntohs(req->length);

  result= length * 4;
  if (remaining < sz_xPSSetStatusMaskReq || (result > remaining))
    return 0;
  
  if (swap)
    {
      req->length = length;
      req->cxid = ntohl(req->cxid);
      req->enableMask = ntohl(req->enableMask);
      req->disableMask = ntohl(req->disableMask);
      req->nextMask = ntohl(req->nextMask);
    }
  
  cpsid = (int)g_hash_table_lookup(conn->cxid_to_cpsid_table, 
				   (gpointer)(req->cxid));
  /* Mask */
  mask = (int)g_hash_table_lookup(conn->cpsid_to_mask_table, 
				  (gpointer)cpsid);
  mask |= req->enableMask;
  mask &= ~req->disableMask;
  g_hash_table_insert(conn->cpsid_to_mask_table, 
		      (gpointer)cpsid,
		      (gpointer)mask);
  if (DGS_DEBUG)
    dps_SetStatusMask_dump("mask: ", mask);
  
  /* Next mask */
  g_hash_table_insert(conn->cpsid_to_nextMask_table, 
		      (gpointer)cpsid,
		      (gpointer)(req->nextMask));
  if (DGS_DEBUG)
    dps_SetStatusMask_dump("next mask: ", req->nextMask);
  
  return result;
}

private void
dps_SetStatusMask_dump(const char * header, int mask)
{
  fprintf(stderr, "%s: \n", header);
  fprintf(stderr, "  PSRUNNINGMASK: %s\n",
	  (mask & PSRUNNINGMASK)?"on":"off");
  fprintf(stderr, "  PSNEEDSINPUTMASK: %s\n",
	  (mask & PSNEEDSINPUTMASK)?"on":"off");
  fprintf(stderr, "  PSZOMBIEMASK: %s\n",
	  (mask & PSZOMBIEMASK)?"on":"off");  
  fprintf(stderr, "  PSFROZENMASK: %s\n",
	  (mask & PSFROZENMASK)?"on":"off");  
}


/* --- dps functions stubs (not implemented) --- */
private void dps_NotifyContext_kill(gsdpsx_conn_info *conn, CONTEXTXID cxid);
private void dps_NotifyContext_interrupt(gsdpsx_conn_info *conn, CONTEXTXID cxid);
private void dps_NotifyContext_unfreeze(gsdpsx_conn_info *conn, CONTEXTXID cxid);
private int
dps_NotifyContext(gsdpsx_conn_info *conn, 
		  unsigned char *pmsg, int remaining,
		  enum scheduling_action * sched_act)
{
  xPSNotifyContextReq *req = (xPSNotifyContextReq *)pmsg;
  short length = req->length;
  int result;
  int swap = util_byteorder_check(conn);
  if (swap)
    length = ntohs(req->length);

  result= length * 4;
  if (remaining < sz_xPSNotifyContextReq
      || (result > remaining))
    result = 0;
  
  if (swap)
    {
      req->length = length;
      req->cxid = ntohl(req->cxid);
    }
  switch(req->notifyType)
    {
    case PSKILL:
      dps_NotifyContext_kill(conn, req->cxid);
      *sched_act = SCHED_ACT_DETACH;
      break;
    case PSINTERRUPT:
      dps_NotifyContext_interrupt(conn, req->cxid);
      break;
    case PSUNFREEZE:
      dps_NotifyContext_unfreeze(conn, req->cxid);
      break;
    }
  return result;
}
private void 
dps_NotifyContext_kill(gsdpsx_conn_info *conn, CONTEXTXID cxid)
{
  int cpsid = (int)g_hash_table_lookup(conn->cxid_to_cpsid_table,
					(gpointer)cxid); 
  /* gs_nulldevice(igs); */
  conn->kill_cpsids_set = 
    g_set_add(conn->kill_cpsids_set, (gpointer)cpsid);
  {
    int next = 0;
    int steady = 1;
    if (!status_send_if_masked(conn, cpsid, PSZOMBIE, next))
      status_send_if_masked(conn, cpsid, PSZOMBIE, steady);
  }
}
private void
dps_NotifyContext_interrupt(gsdpsx_conn_info *conn, CONTEXTXID cxid)
{
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] NotifyContext->interrupt is not implemented\n");
}
private void
dps_NotifyContext_unfreeze(gsdpsx_conn_info *conn, CONTEXTXID cxid)
{
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] NotifyContext->unfreeze is not implemented\n");
}
private int
dps_DestroySpace(gsdpsx_conn_info *conn, unsigned char *pmsg, int remaining)
{
  /* TODO */
  xPSDestroySpaceReq *req = (xPSDestroySpaceReq *)pmsg;
  short length = req->length;
  int result;
  
  if (util_byteorder_check(conn))
    length = ntohs(req->length);
  
  result = length * 4;
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] DestroySpace is not implemented\n");
  if (remaining < sz_xPSDestroySpaceReq
      || (result > remaining))
    result = 0;
  return result;
}

private int
dps_Init         (gsdpsx_conn_info *conn,
		  unsigned char *pmsg, int remaining)
{
  /* TODO */
  xPSInitReq *req = (xPSInitReq *)pmsg;
  short length = req->length;
  int result;

  if (util_byteorder_check(conn))
    length = ntohs(req->length);

  result = length * 4;
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] Init is not implemented\n");
  if (remaining < sz_xPSInitReq
      || (result > remaining))
    result = 0;
  return result;
}

private int
dps_CreateSpace   (gsdpsx_conn_info *conn,
       	    unsigned char *pmsg, int remaining)
{
  /* TODO */
  xPSCreateSpaceReq *req = (xPSCreateSpaceReq *)pmsg;
  short length = req->length;
  int result; 
  
  if (util_byteorder_check(conn))
    length = ntohs(req->length);

  result = length * 4;
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] CreateSpace is not implemented\n");
  if (remaining < sz_xPSCreateSpaceReq
      || (result > remaining))
    result = 0;
  return result;
}

private int
dps_Reset        (gsdpsx_conn_info *conn,
       	   unsigned char *pmsg, int remaining)
{
  /* TODO */
  xPSResetReq *req = (xPSResetReq *)pmsg;
  short length = req->length;
  int result;

  if (util_byteorder_check(conn))
    length = ntohs(req->length);

  result = length * 4;
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] Reset is not implemented\n");
  if (remaining < sz_xPSResetReq
      || (result > remaining))
    result = 0;
  return result;
}

private int
dps_CreateSecureContext(gsdpsx_conn_info *conn,
       		 unsigned char *pmsg, int remaining)
{
  /* TODO */
  xPSCreateSecureContextReq *req = (xPSCreateSecureContextReq *)pmsg;
  short length = req->length;
  int result;

  if (util_byteorder_check(conn))
    length = ntohs(req->length);

  result = length*4;
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] SecureContext is not implemented\n");
  if (remaining < sz_xPSCreateSecureContextReq
      || (result > remaining))
    result = 0;
  return result;
}

private int
dps_NotifyWhenReady(gsdpsx_conn_info *conn,
       	     unsigned char *pmsg, int remaining)
{
  /* TODO */
  xPSNotifyWhenReadyReq *req = (xPSNotifyWhenReadyReq *)pmsg;
  short length = req->length;
  int result;

  if (util_byteorder_check(conn))
    length = ntohs(req->length);

  result = length*4;  
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] NotifyWhenReady is not implemented\n");
  if (remaining < sz_xPSNotifyWhenReadyReq
      || (result > remaining))
    result = 0;
  return result;
}

/* --- dpsX functions stubs (not implemented) --- */
private int 
dpsx_ChangeGC(gsdpsx_conn_info *conn,
	      unsigned char *pmsg, int remaining)
{
  xChangeGCReq *req = (xChangeGCReq*)pmsg;
  int length = req->length;
  int swap = util_byteorder_check(conn);
  int result;
  unsigned int * values;

  if (swap)
    length = ntohs(req->length);
  
  result = length*4;
  if (remaining < sz_xChangeGCReq || 
      (result > remaining))
    return 0;

  if (swap)
    {
      req->length = length;
      req->gc = ntohl(req->gc);
      req->mask = ntohl(req->mask);
    }

  values = (unsigned int*)(req+1);
  if (DGS_DEBUG)
    fprintf(stderr, "[TODO] ChangeGC is not implemented\n");
  return result;
}

/* --- (conn_) connection setting up functions -- */
private void
conn_sock_initialize(void)
{
  
}

private void
conn_sock_finalize(void * sockaddr, enum conn_type type)
{
  if (type == UNIXCONN)
    (void)unlink(((struct sockaddr_un *)sockaddr)->sun_path);
}

private int
conn_hostname_get(char * hostname, enum conn_type type)
{
  if (type == UNIXCONN)
    {
      if (gethostname(hostname, sizeof(char[HOSTNAME_LEN])) ) {
        perror("gethostname: ");
        return -1;
      }
    }
  else if (type == TCPCONN)
    {
      /*! if we use the full hostname, the client library will 
        switch us to UNIX socket */
      /* strcpy(hostname, "localhost"); */
      if (gethostname(hostname, sizeof(char[HOSTNAME_LEN])) ) {
        perror("gethostname: ");
        return -1;
      }
    }

  return 0;
}

private int
conn_server_create(void * sockaddr, enum conn_type type)
{
  int server_socket;
  if (type == UNIXCONN)
    {
      struct stat st;
      (void)mkdir(CSDPS_UNIX_DIR, (S_IRWXU | S_IRWXG | S_IRWXO | S_ISVTX));

      if (stat(CSDPS_UNIX_DIR, &st)) {
        perror("stat: ");
        return -1;
      } else if (!S_ISDIR(st.st_mode)) {
        fprintf(stderr, "Socket directory: %s is not a directory\n",
       	 CSDPS_UNIX_DIR);
        return -1;
      }      
      server_socket = socket(AF_UNIX, SOCK_STREAM, 0);
      if (server_socket < 0) {
        perror("[UNIXCONN] server_socket's socket(): ");
        return -1;
      }
      ((struct sockaddr_un *)sockaddr)->sun_family = AF_UNIX;      
    }
  else if (type == TCPCONN)
    {
      server_socket = socket(AF_INET, SOCK_STREAM, 0);
      if (server_socket < 0) {
        perror("[TCPCONN] server_socket's socket(): ");
        return -1;
      }
      ((struct sockaddr_in *)sockaddr)->sin_family = AF_INET;
      ((struct sockaddr_in *)sockaddr)->sin_addr.s_addr = INADDR_ANY;
    }
  else
    {
      fprintf(stderr,
              "(conn_server_create)Wrong connection type %d\n",
              type);
      server_socket = -1;
    }
  return server_socket;
}

private int
conn_server_bind(int server_socket, 
       	  void * sockaddr, 
       	  int base_port,
       	  enum conn_type type)
{
  int sockaddr_size;
  
  if (type == UNIXCONN)
    {
      sprintf(((struct sockaddr_un *)sockaddr)->sun_path, 
       	 "%s_%d", 
       	 CSDPS_UNIX_PATH, base_port);
        sockaddr_size = (int)sizeof(struct sockaddr_un);
    }
  else if (type == TCPCONN)
    {
      ((struct sockaddr_in *)sockaddr)->sin_port = htons(base_port);
      sockaddr_size = (int)sizeof(struct sockaddr_in);
    }
  else
    {
      fprintf(stderr, 
	      "(conn_server_bind)Wrong connection type %d\n",
	      type);
      return -1;
    }

  if (0 != bind(server_socket,
		(struct sockaddr*)sockaddr,
		sockaddr_size)) 
    {
      fprintf(stderr, 
	      "[%s]Unable to bind a port[%d] for the server socket.\n",
	      util_conn_type_enum2str(type), base_port);
      if (type == UNIXCONN)
	fprintf(stderr,
	       "Check another dgs server and remove a file in %s "
	       "associated with port number: %d.\n",
	       CSDPS_UNIX_DIR, base_port);
      else if (type == TCPCONN)
	fprintf(stderr, "Check another dgs server process.\n");
      return -1;
    }
  else
    return 0;
}

private void
conn_server_close(int server_socket)
{
  close(server_socket);
}

private int
conn_base_port_find(const char * args)
{
  int requested_port = args_get_port(args);
  if (requested_port == 0)
    {
      struct servent *pent;
      pent = getservbyname(DPS_NX_SERV_NAME, "tcp");
      return ( pent ? ntohs(pent->s_port) : CSDPSPORT );
    }
  else
    return requested_port;
}

/* Note: You can inspect X server atom informations by `xprop' 
   command. See man page for more details. */
private int
conn_atom_setup(Display *dpy, 
       	 Window win_unix, int port_unix,
       	 Window win_tcp, int port_tcp)
{
  /* retrieve all the Atoms for the properites we need to set beforehand */
  Atom willingness_atom, open_license_atom;
  int max_willingness = XDPSNX_MAX_WILLINGNESS;
  char license_str[64];
  int transport_info[2];
  char hostname[HOSTNAME_LEN];

  billboard_atom   = XInternAtom(dpy, XDPSNX_BILLBOARD_PROP, FALSE);
  willingness_atom = XInternAtom(dpy, XDPSNX_WILLINGNESS_PROP, FALSE);

  if (!billboard_atom || !willingness_atom) {
    fprintf(stderr, "unable to get X11 Atoms.\n");
    return -1;
  }

  /* put the needed properties into win */
  /* set willingness... XDPSNX_MAX_WILLINGNESS is very willing */
  XChangeProperty(dpy, win_unix,
       	   willingness_atom, XA_INTEGER, 32,
       	   PropModeReplace,
       	   (unsigned char *)&max_willingness, 1);
  XChangeProperty(dpy, win_tcp,
       	   willingness_atom, XA_INTEGER, 32,
       	   PropModeReplace,
       	   (unsigned char *)&max_willingness, 1);


  /* we only support open license, but that should work with any client */
  sprintf(license_str, "%s:%d", LICENSE_METHOD_OPEN, OPEN_LICENSE_VERSION);
  open_license_atom = XInternAtom(dpy, license_str, False);
  XChangeProperty(dpy, win_unix,
       	   XInternAtom(dpy, XDPSNX_LICENSE_METHOD_PROP, False),
       	   XA_ATOM, 32,
       	   PropModeReplace,
       	   (unsigned char *)&open_license_atom, 1);
  XChangeProperty(dpy, win_tcp,
       	   XInternAtom(dpy, XDPSNX_LICENSE_METHOD_PROP, False),
       	   XA_ATOM, 32,
       	   PropModeReplace,
       	   (unsigned char *)&open_license_atom, 1);

  /* set the transport information */
  transport_info[0] = XDPSNX_TRANS_UNIX;
  transport_info[1] = port_unix;
  XChangeProperty(dpy, win_unix,
       	   XInternAtom(dpy, XDPSNX_TRANSPORT_INFO_PROP, False),
       	   XA_INTEGER, 32,
       	   PropModeReplace,
       	   (unsigned char *)&transport_info, 2);

  transport_info[0] = XDPSNX_TRANS_TCP;
  transport_info[1] = port_tcp;
  XChangeProperty(dpy, win_tcp,
       	   XInternAtom(dpy, XDPSNX_TRANSPORT_INFO_PROP, False),
       	   XA_INTEGER, 32,
       	   PropModeReplace,
       	   (unsigned char *)&transport_info, 2);

  /* set the hostname */
  if (conn_hostname_get(hostname, UNIXCONN) == -1) 
    return -1;

  XChangeProperty(dpy, win_unix,
       	   XInternAtom(dpy, XDPSNX_HOST_NAME_PROP, False),
       	   XA_STRING, 8,
       	   PropModeReplace,
       	   (unsigned char *)hostname, strlen(hostname));

  if (conn_hostname_get(hostname, TCPCONN) == -1)
    return -1;
  XChangeProperty(dpy, win_tcp,
       	   XInternAtom(dpy, XDPSNX_HOST_NAME_PROP, False),
       	   XA_STRING, 8,
       	   PropModeReplace,
       	   (unsigned char *)hostname, strlen(hostname));

  /* and append us into the property list */
  /* The order of appending is important.
     We need to prepend win_tcp frist.
     If we prepend win_unix first, remote client
     cannot connect to dgs server 
     -- Masatake */
  XChangeProperty(dpy, DefaultRootWindow(dpy),
       	   billboard_atom, XA_WINDOW, 32,
       	   PropModePrepend,
       	   (unsigned char *)&win_tcp, 1);
  XChangeProperty(dpy, DefaultRootWindow(dpy),
       	   billboard_atom, XA_WINDOW, 32,
       	   PropModePrepend,
       	   (unsigned char *)&win_unix, 1);
  return 0;
}

private int conn_atom_cleanup_window(Window win, 
       			      Window * data, 
       			      int numAgents);

private void
conn_atom_cleanup(void)
{
  if (billboard_atom_needs_cleanup) { 
    /* remove ourselves from the availability list */
    /* we *MUST* do this after the ConnSetup, because the client has
       the display locked at this time, it will deadlock X if we
       XGrabServer here */
    Window *data;
    Atom returnType;
    unsigned long numAgents, bytesLeft;
    int format, i;

    billboard_atom_needs_cleanup = 0;
    fprintf(stderr, "attempting to grab server.\n");
    fflush(stderr);
    XGrabServer(shared_dpy);
    fprintf(stderr, "grabbed server.\n");
    fflush(stderr);
    i = XGetWindowProperty(shared_dpy, 
			   DefaultRootWindow(shared_dpy),
			   billboard_atom,
			   0, /* starting offset */
			   ((((unsigned long) -1) >> 1) / 4), /* maximum to return */
			   False, /* don't delete */
			   XA_WINDOW,
			   &returnType,
			   &format,
			   &numAgents,
			   &bytesLeft,
			   (unsigned char **)&data);
    if (i == Success && returnType == XA_WINDOW) 
      {
        if (numAgents <= 1) 
          {
            fprintf(stderr, "deleteing property\n");
            XDeleteProperty(shared_dpy,
			    DefaultRootWindow(shared_dpy),
			    billboard_atom);
          } 
        else 
          {
            int unix_remove, tcp_remove;
            fprintf(stderr, "found %lu agents\n", numAgents);
            /* remove the agent from the list */
            unix_remove = conn_atom_cleanup_window(server_win_unix, 
						   data, numAgents--);
            tcp_remove = conn_atom_cleanup_window(server_win_tcp, 
						  data, numAgents--);
            if ((tcp_remove == 0) || (unix_remove == 0))
              {
		XChangeProperty(shared_dpy, DefaultRootWindow(shared_dpy),
				billboard_atom, XA_WINDOW, 32,
				PropModeReplace,
				(unsigned char *)data, numAgents);
		fprintf(stderr, "Removed agent from list\n");
              }
            else
              {
		fprintf(stderr, "Cannot find our window in the atom list\n");
		fprintf(stderr, "Something wrong\n");
              }
          }
      }
    XUngrabServer(shared_dpy);
    XFlush(shared_dpy);
    fprintf(stderr, "done with property munging\n");
    fflush(stderr);
  } /* end removing property */  
}

private int
conn_atom_cleanup_window(Window win, Window * data, int numAgents)
{
  int i;
  int found = 0;
  for (i = 0; i < numAgents; i++) 
    {
      if (data[i] == win) 
        {
          fprintf(stderr, "found our window at %d\n", i);
          numAgents--;
	  found = 1;
          break;
        }
    }
  if (found == 0)
    return -1;				     /* Not Found */

  /* Pack the data */
  for (i++; i < numAgents; i++) 
    data[i-1] = data[i];
  return 0;
}
private int
conn_insock_accept(int server_socket, enum conn_type type)
{
  void * in_sockaddr;
  struct sockaddr_un in_sockaddr_un;
  struct sockaddr_in in_sockaddr_in;
  int in_sock, addrlen;

  if (type == UNIXCONN)
    {
      addrlen = sizeof(struct sockaddr_un);
      in_sockaddr = &in_sockaddr_un;
    }
  else if (type == TCPCONN)
    {
      addrlen = sizeof(struct sockaddr_in);
      in_sockaddr = &in_sockaddr_in;
    }
  in_sock = accept(server_socket, (struct sockaddr *)in_sockaddr, &addrlen);

  if (in_sock < 0)
    {
      perror("accept server socket:");
      return - 1;
    }
  else
    return in_sock;
}

private void
conn_insock_close(int insock)
{
  close(insock);
}

/* --- args parser --- */

private const char *
args_get(void)
{
  return getenv(args_key);
}

/* Retrun 0 if the port is not specified */
private int
args_get_port(const char * args)
{
  const char * unix_str = strstr(args, "unix/");
  const char * tcp_str = strstr(args, "tcp/");
  const char * decnet_str = strstr(args, "decnet/");
  const char * tmp;
  const char * port_str;
  int port;

  if (decnet_str != NULL)
    {
      fprintf(stderr, "decnet is not supported.\n");
      return 0;
    }
  else if (unix_str != NULL)
    tmp = strchr(unix_str, '/');
  else if (tcp_str != NULL)
    tmp = strchr(tcp_str, '/');
  else
    return 0;

  if ((!tmp) || (tmp[1] == '\0'))
    return 0;

  port_str = tmp+1;
  port = atoi(port_str);
  if (DGS_DEBUG)
    fprintf(stderr, 
	    "Passed port[%d] via environment variable.\n",
	    port);  

  return port;
}

private int 
args_get_int(const char * args, const char * key)
{
  int value;
  char * tmp;
  char extended_key[64];
  char format[64]; 
  
  if (args == 0 || key == 0)
    return 0;
  
  sprintf(extended_key, "--%s=", key);
  tmp  = strstr(args, extended_key);
  if (tmp == 0)
    return 0;

  sprintf(format, "--%s=%s", key, "%d");
  if (sscanf(tmp, format, &value) != 1)
    return 0;
  else
    return value;
}
private int
args_get_debug(const char * args)
{
  const char * key = "debug";
  return args_get_int(args, key);
}

private int
args_get_match(const char * args, const char * key)
{
  char extended_key[64];
  char * tmp;
  sprintf(extended_key, "--%s", key);
  tmp = strstr(args, extended_key);

  if (tmp == 0)
    return 0;
  else
    return 1;
}

/* --- buffer functions --- */

private gsdpsx_conn_buf *
buf_init(void)
{
    static gsdpsx_conn_buf conn_buf;
    conn_buf.buf = gs_malloc(BUFLEN, 1, "buf_init");
    if (!(conn_buf.buf))
	{
	    fprintf(stderr, "Memroy allocation is failed.\n");
	    return 0;
	}
    conn_buf.curpos = conn_buf.buf;
    return &conn_buf;
}

private void
buf_reset(gsdpsx_conn_buf * buf)
{
  buf->curpos = buf->buf;
  buf->len = 0;
  buf->isfrag = 0;
}

private int
buf_recv(gsdpsx_conn_buf * buf, gsdpsx_conn_info * conn)
{
  return recv(conn->sock, buf->curpos, BUFLEN - (buf->curpos - buf->buf), 0);
}

private void
buf_input_set(gsdpsx_conn_buf * buf, unsigned char * start, int length)
{
    buf->inputstart = start;
    buf->inputend = buf->inputstart + length;
}
private int
buf_input_length(gsdpsx_conn_buf * buf)
{
    return (buf->inputend - buf->inputstart);
}

private void
buf_input_forward(gsdpsx_conn_buf * buf, stream_cursor_write *pw)
{
  int tocopy;
  int stream_size = pw->limit - pw->ptr;
  
  {
    const int dump_if_op = 0;
    if (dump_if_op)
      buf_dump(buf, stream_size, dump_if_op);
  }

  tocopy = min(buf_input_length(buf), stream_size);
  memmove(pw->ptr+1, buf->inputstart, tocopy);
  pw->ptr += tocopy;    
  buf->inputstart += tocopy;
  buf_input_sanity_check(buf);

}

private void
buf_input_sanity_check(gsdpsx_conn_buf * buf)
{
    if (buf->inputstart >= buf->inputend)
      buf_input_set(buf, NULL, 0);
}

private void 
buf_dump(gsdpsx_conn_buf * buf, int length, unsigned char op)
{
  int i;
  int found = 0;
  for (i = 0; i < length; i++)
    {
      if (buf->inputstart[i] == op)
	found = 1;
    }
  if (found)
    {
      fprintf(stderr, "bin dump start:[%d]{", length);
      for (i = 0; i < length; i++)
	fprintf(stderr, "%d,", buf->inputstart[i]);
      fprintf(stderr, "};\n");
    }
}
private void
buf_pack(gsdpsx_conn_buf * buf, unsigned char *cur, int len)
{
  dprintf1("Fragment of %d bytes\n", len);
  memmove(buf->buf, cur, len);
  buf_move_cursor(buf, buf->buf + len, len);
  buf->isfrag = 1;
}

private void
buf_move_cursor(gsdpsx_conn_buf * buf, unsigned char *cur, int len)
{
  buf->curpos = cur;
  buf->len = len;
}

private void
buf_finalize(gsdpsx_conn_buf * buf)
{
    gs_free(buf->buf, 1, sizeof(BUFLEN), "buf_finalize");
    buf->buf = 0;
}

/* --- debug facilities --- */

/* For protocol debugging */
const private 
char * util_x_ps_request_table [] =
{
  "WrongPSRequest",
  "X_PSInit",			/* 1 */
  "X_PSCreateContext",		/* 2, Has stub*/
  "X_PSCreateSpace",		/* 3 */
  "X_PSGiveInput",		/* 4, Has stub */
  "X_PSGetStatus",		/* 5 */
  "X_PSDestroySpace",		/* 6, Has stub */
  "X_PSReset",			/* 7 */
  "X_PSNotifyContext",		/* 8, Has stub */
  "X_PSCreateContextFromID",	/* 9 */
  "X_PSXIDFromContext",		/* 10 */
  "X_PSContextFromXID",		/* 11 */
  "X_PSSetStatusMask",		/* 12,  Has stub */
  "X_PSCreateSecureContext",	/* 13, L2-DPS/PROTO 9 addition */
  "X_PSNotifyWhenReady",	/* 14, L2-DPS/PROTO 9 addition */    
};

private const char *
util_x_ps_request_enum2str(int key)
{
    if ((1 > key) && (key > PSLASTREQUEST))
        key = 0;
    return util_x_ps_request_table[key];   
}

const private 
char * util_x_cap_request_table [] =
{
    "WrongCAPRequest",
    "X_CAPFlushAgent",		/* 1 */
    "X_CAPNotify",		/* 2 */
    "X_CAPSetArg",		/* 3 */
};

private const char *
util_x_cap_request_enum2str(int key)
{
  if ((1 > key) && (key > X_CAPSetArg))
    key = 0;
  return util_x_cap_request_table[key];
}

private const char *
util_conn_type_enum2str(enum conn_type type)
{
  if (type == UNIXCONN)
    return "UNIXCONN";
  else if (type == TCPCONN)
    return "TCPCONN";
  else
    return "UNKNOWN CONN";
}

private const
char * util_status_enum2str(int key)
{
  switch (key)
    {
    case PSSTATUSERROR:
      return "PSSTATUSERROR";
    case PSRUNNING:
      return "PSRUNNING";
    case PSNEEDSINPUT:
	return "PSNEEDSINPUT";
    case PSZOMBIE:
      return "PSZOMBIE";
    case PSFROZEN:
      return "PSFROZEN";
    default:
      return "Unknown status";
    }
}

private const char *
util_signal_int2str(int sig)
{
  switch (sig)
    {
    case 0: 
      return "EOF";
    case SIGHUP:
      return "SIGHUP";
    case SIGTERM:
      return "SIGTERM";
    case SIGINT:
      return "SIGINT";
    case SIGCHLD:
      return "SIGCHLD";
    case SIGSEGV:
      return "SIGSEGV";
    default:
      return "Unknown";
    }
}

private void
util_print_pid(int sec)
{
  int i;

  if (sec < 2)
    return;

  fprintf(stderr, "attach %d\n", getpid());
  for (i = 0; i < sec; i++)
    {
      sleep(1);
      fprintf(stderr, "."); 
    }
  fprintf(stderr, "\n"); 
}

/* Return 1 if the htons or htonl invocation is needed. */
private int
util_byteorder_check(gsdpsx_conn_info *conn)
{
  if ((shared_byteorder == 'l') && (conn->byteorder == 'B'))
    return 1;
  else
    return 0;
}

/* --- status --- */
private int status_check_mask(int status, int mask);
private int status_to_mask(int status);
private void status_send(gsdpsx_conn_info *conn,
			 int cpsid,
			 int status);
private int
status_send_if_masked(gsdpsx_conn_info *conn,
		      int cpsid,
		      int status,
		      int steady1_or_next0)
{
  GHashTable * table = (steady1_or_next0)?
    conn->cpsid_to_mask_table:
    conn->cpsid_to_nextMask_table;
  int mask = (int)g_hash_table_lookup(table, (gpointer)cpsid);
  int result = 0;
  if (status_check_mask(status, mask))
    {
      result = 1;
      status_send(conn, cpsid, status);
      if (steady1_or_next0 == 0)
	{
	  mask &= ~status_to_mask(status);
	  g_hash_table_insert(table,
			      (gpointer)cpsid,
			      (gpointer)mask);
	}
      if (DGS_DEBUG)
	fprintf(stderr, "Send status event: %s\n",
		util_status_enum2str(status));
    }
  return result;
}

private void
status_send(gsdpsx_conn_info *conn, int cpsid, int status)
{
  XEvent event;
  DPSCAPStatusEvent * status_event;
  static Atom statusAtom = 0;
  gx_device_X * xdev = (gx_device_X *)gs_currentdevice(igs);
  CONTEXTXID cxid = 
    (CONTEXTXID)g_hash_table_lookup(conn->cpsid_to_cxid_table,
				    (void *)cpsid);
  
  if (!xdev || !xdev->dpy)
    return;

  if (!statusAtom)
    statusAtom = XInternAtom(xdev->dpy, DPSCAP_TYPE_PSSTATUS, False);  
  
  event.xclient.type = ClientMessage;
  event.xclient.serial = 0;
  event.xclient.send_event = True;
  event.xclient.message_type = statusAtom;
  event.xclient.format = 8;	/* ??? */
  status_event = (DPSCAPStatusEvent*)event.xclient.data.l;

  status_event->status = status;
  if (util_byteorder_check(conn))
    {
      status_event->cxid = htonl(cxid);
      status_event->sequenceNumber = htons(0);
    }
  else
    {
      status_event->cxid = cxid;
      status_event->sequenceNumber = 0;
    }
  event.xclient.display = xdev->dpy;
  event.xclient.window = conn->clientWindow;
  XSendEvent(event.xclient.display, event.xclient.window, False, 0, &event);
  XSync(xdev->dpy, False);
}

private int
status_check_mask(int status, int mask)
{
  return (status_to_mask(status) & mask);
}
private int
status_to_mask(int status)
{
  return 1 << (status - 1);
}

/* --- Context switching via  time slice control --- */
private int ticks_left;

void
ticks_left_set(int value)
{
  ticks_left = value;
}

void
ticks_left_zero(void)
{
  ticks_left_set(0);
}

int
ticks_left_decrement(void)
{
  return --ticks_left;
}


int
ticks_left_value(void)
{
  return ticks_left;
}

int *
ticks_left_refer(void)
{
  return &ticks_left;
}

/* --- context init parameter  --- */
private void
ctx_init_parameter_initialize(struct ctx_init_parameter  * parameter)
{
  parameter->cxid = 0;
  parameter->drawable = 0;
  parameter->offset_x = 0;
  parameter->offset_y = 0;
}
private int
ctx_init_parameter_install(struct ctx_init_parameter  * parameter)
{
  /* imemory, igs   */
  float xppp, yppp;
  gx_device_X *new_dev;
  gs_matrix *initial_matrix;
  int code;

  if (conn_info.primary_device == NULL)
    {
      code = gs_copydevice((gx_device**)&new_dev,
			   (gx_device*)&gs_x11_device,
			   imemory);
      conn_info.primary_device = new_dev;
    }
  else
      code = gs_copydevice((gx_device**)&new_dev,
			   (gx_device*)conn_info.primary_device,
			   imemory);

  if (code < 0)
    {
      fprintf(stderr,"Fail to copy device\n");
      return code;
    }
  new_dev->owner = gs_x_dpsagent;
  new_dev->pwin  = conn_info.clientWindow;

  code = gs_setdevice(igs, (gx_device*)new_dev);
  if (code < 0)
    {
      fprintf(stderr,"Fail to set new device\n");
      return code;
    }

  gdev_x_set_drawable(new_dev, parameter->drawable);
  initial_matrix = xdev_initial_matrix_inline(new_dev);
  xppp = new_dev->x_pixels_per_inch / 72.0;
  yppp = new_dev->y_pixels_per_inch / 72.0;
  initial_matrix->xx = xppp;
  initial_matrix->xy = 0.0;
  initial_matrix->yx = 0.0;
  initial_matrix->yy = -yppp;
  initial_matrix->tx = 0.0;
  initial_matrix->ty = 0.0;
  gsdpsx_set_offset(new_dev, parameter->offset_x, 0);
  gsdpsx_set_offset(new_dev, parameter->offset_y, 1);
  gs_initmatrix(igs);
  gs_initclip(igs);
  return 0;
}

int
gsdpsx_get_offset(gx_device_X * dev, int x0_or_y1)
{
  if (x0_or_y1 == 0)		/* X */
    return (int)(dev_x_offset(dev) * dev->x_pixels_per_inch);
  else				/* Y */
    return (int)(dev_y_offset(dev) * dev->y_pixels_per_inch);
}

void
gsdpsx_set_offset(gx_device_X * dev, int offset, int x0_or_y1)
{
  if (x0_or_y1 == 0)		/* X */
    dev->Margins[0] = (float)offset * 
      dev->MarginsHWResolution[0] / dev->x_pixels_per_inch; 
  else				/* Y */
    dev->Margins[1] = (float)offset * 
      dev->MarginsHWResolution[1] / dev->y_pixels_per_inch; 
}


/* --- GSet --- */
private GSet * 
g_set_add(GSet * set, gpointer data)
{
  return g_slist_append(set, data);
}
private GSet *
g_set_remove(GSet * set, gpointer data)
{
  return g_slist_remove(set, data);
}
private gpointer
g_set_member (GSet *set, gpointer   data)
{
  return g_slist_find(set, data);
}
private void
g_set_foreach (GSet *set, GFunc func, gpointer   user_data)
{
  g_slist_foreach(set, func, user_data);
}
private guint
g_set_count  (GSet * set)
{
  return g_slist_length(set);
}
