/* glpapi/glp_first_coef.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_first_coef - find the first constraint coefficient.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_first_coef(int what);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_first_coef routine finds the first
-- constraint coefficient placed in the current row and makes it be the
-- current coefficient. At the same time the routine also makes the
-- column, which corresponds to the found coefficient, be the current
-- column (the current row remains unchanged).
--
-- If what = GLP_COL, the glp_first_coef routine finds the first
-- constraint coefficient placed in the current column and makes it be
-- the current coefficient. At the same time the routine also makes the
-- row, which corresponds to the found coefficient, be the current row
-- (the current column remains unchanged).
--
-- The ordering of constraint coefficients can be changed by GLPK API
-- routines, therefore the application program shouldn't rely on some
-- particular ordering (except cases, when the particular ordering is
-- explictly declared).
--
-- *Complexity*
--
-- This operation has time complexity O(1).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row is undefined or empty (in case of GLP_ROW), or
--     the current column is undefined or empty (in case of GLP_COL);
-- 2 - the parameter what is invalid. */

int glp_first_coef(int what)
{     if (what == GLP_ROW)
      {  if (glp->this_row == NULL) return 1;
         glp->this_coef = glp->this_row->ptr;
         if (glp->this_coef == NULL) return 1;
         glp->this_col = glp->this_coef->col;
      }
      else if (what == GLP_COL)
      {  if (glp->this_col == NULL) return 1;
         glp->this_coef = glp->this_col->ptr;
         if (glp->this_coef == NULL) return 1;
         glp->this_row = glp->this_coef->row;
      }
      else
      {  /* invalid parameter */
         return 2;
      }
      return 0;
}

/* eof */
