/* glpapi/glp_get_activity.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_get_activity - get activity of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_activity(int what, int *tagx, double *valx, double *dx);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_get_activity routine stores the status,
-- the primal activity, and the dual activity of the current row (i.e.
-- of the corresponding auxiliary variable) to locations, which
-- parameters tagx, valx, and dx point to, respectively.
--
-- If what = GLP_COL, the glp_get_activity routine stores the same
-- information, but for the current column (i.e. for the corresponding
-- structural variable).
--
-- Information reported by the glp_get_activity routine is determined
-- by the final simplex table found by the solver. This information has
-- the meaning only if the problem has been successfully solved.
--
-- The status of the variable may be one of the following:
--
-- GLP_BS - the variable is basis;
-- GLP_NL - the variable is non-basis and placed on its lower bound
--          (only for variables of GLP_LO and GLP_DB types);
-- GLP_NU - the variable is non-basis and placed on its upper bound
--          (only for variables of GLP_UP and GLP_DB types);
-- GLP_NF - the free variable is non-basis (only for variables of
--          GLP_FR type);
-- GLP_NS - the fixed variable is non-basis (only for variables of
--          GLP_FX type).
--
-- The primal activity is the computed value of the variable. (Note that
-- value of any free non-basis variable is always equal to zero.)
--
-- The dual activity is the computed reduced cost (marginal value) of
-- the variable.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid. */

int glp_get_activity(int what, int *tagx, double *valx, double *dx)
{     GLPITEM *item;
      if (what == GLP_ROW)
         item = glp->this_row;
      else if (what == GLP_COL)
         item = glp->this_col;
      else
      {  /* invalid parameter */
         return 2;
      }
      if (item == NULL) return 1;
      if (tagx != NULL) *tagx = item->tagx;
      if (valx != NULL) *valx = item->valx;
      if (dx != NULL) *dx = item->dx;
      return 0;
}

/* eof */
