/* glpapi/glp_print_sol.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include "glpk.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- glp_print_sol - write problem solution using printable format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_print_sol(char *fname);
--
-- *Description*
--
-- The glp_print_sol routine writes the problem solution using printable
-- format to the text file, whose name is the character string fname.
--
-- Information reported by the routine corresponds to the final simplex
-- table found by the solver. This information is intended mainly for
-- the visual analysis and has the meaning only if the problem has been
-- successfully solved.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_print_sol(char *fname)
{     FILE *fp;
      int kase;
      print("glp_print_sol: writing problem solution to `%s'...",
         fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_print_sol: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* problem name */
      {  char name[GLP_MAX_NAME+1];
         glp_get_cpar("problem", name);
         fprintf(fp, "%-12s%s\n", "Problem:", name);
      }
      /* number of rows (auxiliary variables) */
      {  int nr;
         glp_get_ipar("nr", &nr);
         fprintf(fp, "%-12s%d\n", "Rows:", nr);
      }
      /* number of columns (structural variables) */
      {  int nc;
         glp_get_ipar("nc", &nc);
         fprintf(fp, "%-12s%d\n", "Columns:", nc);
      }
      /* number of non-zeros (constraint coefficients) */
      {  int nz;
         glp_get_ipar("nz", &nz);
         fprintf(fp, "%-12s%d\n", "Non-zeros:", nz);
      }
      /* objective function */
      {  char name[GLP_MAX_NAME+1];
         fprintf(fp, "%-12s", "Objective:");
         glp_get_cpar("obj_row", name);
         if (name[0] == '\0')
            fprintf(fp, "UNDEFINED\n");
         else
         {  int dir;
            double val;
            glp_get_ipar("obj_dir", &dir);
            glp_find_item(GLP_ROW, name);
            glp_get_activity(GLP_ROW, NULL, &val, NULL);
            fprintf(fp, "%s = %.6g %s\n", name, val,
               dir == GLP_MIN  ? "(MINimization)" :
               dir == GLP_MAX  ? "(MAXimization)" : "");
         }
      }
      /* problem status */
      {  int status;
         glp_get_ipar("status", &status);
         fprintf(fp, "%-12s%s\n", "Status:",
            status == GLP_UNDEF  ? "UNDEFINED" :
            status == GLP_OPT    ? "OPTIMAL" :
            status == GLP_FEAS   ? "FEASIBLE" :
            status == GLP_INFEAS ? "INFEASIBLE (INTERMEDIATE)" :
            status == GLP_NOFEAS ? "INFEASIBLE (FINAL)" :
            status == GLP_UNBND  ? "UNBOUNDED" : "???");
      }
      /* main output */
      for (kase = 0; kase <= 1; kase++)
      {  int what = kase == 0 ? GLP_ROW : GLP_COL, numb = 0, ret;
         char name[GLP_MAX_NAME+1];
         fprintf(fp, "\n");
         fprintf(fp, "Number %-12s St   Activity     Lower bound   Uppe"
            "r bound    Marginal\n",
             what == GLP_ROW ? "  Row name" : "Column name");
         fprintf(fp, "------ ------------ -- ------------- ------------"
            "- ------------- -------------\n");
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int type, tagx;
            double lb, ub, valx, dx;
            /* row/column sequential number */
            numb++;
            fprintf(fp, "%6d ", numb);
            /* row column/name */
            glp_get_name(what, name);
            if (strlen(name) <= 12)
               fprintf(fp, "%-12s ", name);
            else
               fprintf(fp, "%s\n%20s", name, "");
            /* row/column ... */
            glp_get_bounds(what, &type, &lb, &ub);
            glp_get_activity(what, &tagx, &valx, &dx);
            /* row/column status */
            if (tagx == GLP_BS)
            {  if (type == GLP_LO && valx < lb ||
                   type == GLP_UP && valx > ub ||
                   type == GLP_DB && valx < lb ||
                   type == GLP_DB && valx > ub ||
                   type == GLP_FX && valx != lb)
                  fprintf(fp, "B* ");
               else
                  fprintf(fp, "B  ");
            }
            else if (tagx == GLP_NL)
               fprintf(fp, "NL ");
            else if (tagx == GLP_NU)
               fprintf(fp, "NU ");
            else if (tagx == GLP_NF)
               fprintf(fp, "NF ");
            else if (tagx == GLP_NS)
               fprintf(fp, "NS ");
            else
               fprintf(fp, "??");
            /* row/column primal activity */
            fprintf(fp, "%13.6g ", valx);
            /* row/column lower bound */
            if (type == GLP_LO || type == GLP_DB || type == GLP_FX)
               fprintf(fp, "%13.6g ", lb);
            else
               fprintf(fp, "%13s ", "");
            /* row/column upper bound */
            if (type == GLP_UP || type == GLP_DB)
               fprintf(fp, "%13.6g ", ub);
            else if (type == GLP_FX)
               fprintf(fp, "%13s ", "=");
            else
               fprintf(fp, "%13s ", "");
            /* row/column dual activity */
            if (tagx != GLP_BS)
            {  if (dx == 0.0)
                  fprintf(fp, "%13s", "< eps");
               else
                  fprintf(fp, "%13.6g", dx);
            }
            fprintf(fp, "\n");
         }
      }
      fprintf(fp, "\n");
      fprintf(fp, "End of output\n");
      fflush(fp);
      if (ferror(fp))
      {  error("glp_print_sol: can't write to `%s' - %s", fname,
            strerror(errno));
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
