/* glpapi/glp_read_mps.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <float.h>
#include <math.h>
#include <string.h>
#include "glpk.h"
#include "glpmps.h"

/*----------------------------------------------------------------------
-- glp_read_mps - read problem data using MPS format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_read_mps(char *fname)
--
-- *Description*
--
-- The glp_read_mps routine reads LP problem data using MPS format from
-- the text file, whose name is the character string name, into the
-- workspace.
--
-- As a rule the workspace should be empty before a call to the
-- glp_read_mps routine, i.e. the workspace should contain no rows and
-- no columns.
--
-- Detailed description of the MPS format can be found, for example,
-- in the following book:
--
-- B.A.Murtagh. Advanced Linear Programming: Computation and Practice.
-- McGraw-Hill, 1981.
--
-- *Control parameters*
--
-- The behavior of the glp_read_mps routine depends on the following
-- control parameters:
--
-- mps_obj_name (the name of row that specifies the objective function);
-- mps_rhs_name (the name of right-hand side vector);
-- mps_rng_name (the name of range vector);
-- mps_bnd_name (the name of bound vector).
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_read_mps(char *fname)
{     MPS *mps;
      MPSROW *row; MPSCOL *col; MPSCQE *cqe; MPSBQE *bqe;
      int m, n, i, j, v;
      char obj[8+1], rhs[8+1], rng[8+1], bnd[8+1];
      glp_get_cpar("mps_obj_name", obj);
      glp_get_cpar("mps_rhs_name", rhs);
      glp_get_cpar("mps_rng_name", rng);
      glp_get_cpar("mps_bnd_name", bnd);
      /* load MPS data block */
      mps = load_mps(fname);
      if (mps == NULL) goto fail;
      m = mps->n_row;
      n = mps->n_col;
      /* set problem name */
      glp_set_cpar("problem", mps->name);
      /* process ROWS section */
      for (i = 1; i <= m; i++)
      {  int type;
         row = mps->row[i];
         if (glp_create_item(GLP_ROW, row->name))
         {  error("glp_read_mps: error on creating row `%s'",
               row->name);
            goto fail;
         }
         if (strcmp(row->type, "N") == 0)
            type = GLP_FR;
         else if (strcmp(row->type, "L") == 0)
            type = GLP_UP;
         else if (strcmp(row->type, "G") == 0)
            type = GLP_LO;
         else if (strcmp(row->type, "E") == 0)
            type = GLP_FX;
         else
         {  error("glp_read_mps: row `%s' has unknown type `%s'",
               row->name, row->type);
            goto fail;
         }
         glp_set_bounds(GLP_ROW, type, 0.0, 0.0);
      }
      /* process COLUMNS section */
      for (j = 1; j <= n; j++)
      {  col = mps->col[j];
         if (glp_create_item(GLP_COL, col->name))
         {  error("glp_read_mps: error on creating column `%s'",
               col->name);
            goto fail;
         }
         glp_set_bounds(GLP_COL, GLP_LO, 0.0, 0.0);
         for (cqe = col->ptr; cqe != NULL; cqe = cqe->next)
         {  glp_find_item(GLP_ROW, mps->row[cqe->ind]->name);
            glp_new_coef(cqe->val);
         }
      }
      /* set the objective function */
      glp_set_ipar("obj_dir", GLP_ANY);
      glp_set_cpar("obj_row", "");
      for (i = 1; i <= m; i++)
      {  row = mps->row[i];
         if (obj[0] == '\0' && strcmp(row->type, "N") == 0 ||
             obj[0] != '\0' && strcmp(row->name, obj) == 0)
         {  glp_set_ipar("obj_dir", GLP_MIN);
            glp_set_cpar("obj_row", row->name);
            break;
         }
      }
      if (obj != '\0' && i > m)
      {  error("glp_read_mps: objective function row `%s' not found",
            obj);
         goto fail;
      }
      /* process RHS section */
      if (rhs == NULL || rhs[0] == '\0')
         v = (mps->n_rhs == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_rhs; v++)
            if (strcmp(mps->rhs[v]->name, rhs) == 0) break;
         if (v > mps->n_rhs)
         {  error("glp_read_mps: right-hand side vector `%s' not found",
               rhs);
            goto fail;
         }
      }
      if (v > 0)
      {  for (cqe = mps->rhs[v]->ptr; cqe != NULL; cqe = cqe->next)
         {  int type = GLP_FR;
            double lb = 0.0, ub = 0.0;
            glp_find_item(GLP_ROW, mps->row[cqe->ind]->name);
            glp_get_bounds(GLP_ROW, &type, NULL, NULL);
            switch (type)
            {  case GLP_FR:
                  break;
               case GLP_LO:
                  lb = cqe->val;
                  break;
               case GLP_UP:
                  ub = cqe->val;
                  break;
               case GLP_FX:
                  lb = ub = cqe->val;
                  break;
               default:
                  assert(type != type);
            }
            glp_set_bounds(GLP_ROW, type, lb, ub);
         }
      }
      /* process RANGES section */
      if (rng == NULL || rng[0] == '\0')
         v = (mps->n_rng == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_rng; v++)
            if (strcmp(mps->rng[v]->name, rng) == 0) break;
         if (v > mps->n_rng)
         {  error("glp_read_mps: range vector `%s' not found", rng);
            goto fail;
         }
      }
      if (v > 0)
      {  for (cqe = mps->rng[v]->ptr; cqe != NULL; cqe = cqe->next)
         {  int type = GLP_FR;
            double lb = 0.0, ub = 0.0;
            glp_find_item(GLP_ROW, mps->row[cqe->ind]->name);
            glp_get_bounds(GLP_ROW, &type, &lb, &ub);
            switch (type)
            {  case GLP_FR:
                  error("glp_read_mps: range vector entry refers to row"
                     " `%s' of N type", mps->row[cqe->ind]->name);
                  goto fail;
               case GLP_LO:
                  ub = lb + fabs(cqe->val);
                  break;
               case GLP_UP:
                  lb = ub - fabs(cqe->val);
                  break;
               case GLP_FX:
                  if (cqe->val >= 0.0)
                     ub += fabs(cqe->val);
                  else
                     lb -= fabs(cqe->val);
                  break;
               default:
                  assert(type != type);
            }
            glp_set_bounds(GLP_ROW, lb == ub ? GLP_FX : GLP_DB, lb, ub);
         }
      }
      /* process BOUNDS section */
      if (bnd == NULL || bnd[0] == '\0')
         v = (mps->n_bnd == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_bnd; v++)
            if (strcmp(mps->bnd[v]->name, bnd) == 0) break;
         if (v > mps->n_bnd)
         {  error("glp_read_mps: bound vector `%s' not found", bnd);
            goto fail;
         }
      }
      if (v > 0)
      {  for (bqe = mps->bnd[v]->ptr; bqe != NULL; bqe = bqe->next)
         {  int type = GLP_FX;
            double lb = 0.0, ub = 0.0;
            glp_find_item(GLP_COL, mps->col[bqe->ind]->name);
            glp_get_bounds(GLP_COL, &type, &lb, &ub);
            if (type == GLP_FR || type == GLP_UP) lb = -DBL_MAX;
            if (type == GLP_FR || type == GLP_LO) ub = +DBL_MAX;
            if (strcmp(bqe->type, "LO") == 0)
               lb = bqe->val;
            else if (strcmp(bqe->type, "UP") == 0)
               ub = bqe->val;
            else if (strcmp(bqe->type, "FX") == 0)
               lb = ub = bqe->val;
            else if (strcmp(bqe->type, "FR") == 0)
               lb = -DBL_MAX, ub = +DBL_MAX;
            else if (strcmp(bqe->type, "MI") == 0)
               lb = -DBL_MAX;
            else if (strcmp(bqe->type, "PL") == 0)
               ub = +DBL_MAX;
            else
            {  error("glp_read_mps: bound vector entry for column `%s' "
                  "has unknown type `%s'",
                  mps->col[bqe->ind]->name, bqe->type);
               goto fail;
            }
            if (lb == -DBL_MAX && ub == +DBL_MAX)
               type = GLP_FR;
            else if (ub == +DBL_MAX)
               type = GLP_LO;
            else if (lb == -DBL_MAX)
               type = GLP_UP;
            else if (lb != ub)
               type = GLP_DB;
            else
               type = GLP_FX;
            glp_set_bounds(GLP_COL, type, lb, ub);
         }
      }
      /* free MPS data block */
      free_mps(mps);
      /* returns to the calling program */
      return 0;
fail: /* the operation failed */
      if (mps != NULL) free_mps(mps);
      return 1;
}

/* eof */
