/* glpimg/save_as_bmp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <limits.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include "glpimg.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- save_as_bmp - write raster image using Windows Bitmap format.
--
-- *Synopsis*
--
-- #include "glpimg.h"
-- int save_as_bmp(IMG *img, char *fname);
--
-- *Description*
--
-- The save_as_bmp routine writes the raster image using uncompressed
-- Windows Bitmap format to the binary file whose name is the character
-- string fname.
--
-- *Returns*
--
-- If the operation completed successfully, the routine returns zero.
-- Otherwise the routine returns non-zero. */

static void put_byte(FILE *fp, int val)
{     unsigned char b = (unsigned char)val;
      fwrite(&b, sizeof(char), 1, fp);
      return;
}

static void put_word(FILE *fp, int val) /* big endian */
{     put_byte(fp, val);
      put_byte(fp, val >> 8);
      return;
}

static void put_dword(FILE *fp, int val) /* big endian */
{     put_word(fp, val);
      put_word(fp, val >> 16);
      return;
}

int save_as_bmp(IMG *img, char *fname)
{     FILE *fp;
      int offset, bmsize, bitcnt;
      fp = fopen(fname, "wb");
      if (fp == NULL)
      {  error("save_as_bmp: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      switch (img->type)
      {  case IMG_2: /* 2 colors (1 bit per pixel) */
            offset = 14 + 40 + 2 * 4;
            bmsize = (1 * img->xsize + 31) / 32;
            bitcnt = 1;
            break;
         case IMG_4: /* 4 colors (2 bits per pixel) */
         case IMG_16: /* 16 colors (4 bits per pixel) */
            offset = 14 + 40 + 16 * 4;
            bmsize = (4 * img->xsize + 31) / 32;
            bitcnt = 4;
            break;
         case IMG_256: /* 256 colors (8 bits per pixel) */
            offset = 14 + 40 + 256 * 4;
            bmsize = (8 * img->xsize + 31) / 32;
            bitcnt = 8;
            break;
         default:
            assert(img->type != img->type);
      }
      /* struct BMPFILEHEADER */
      /* UINT bfType */
      put_byte(fp, 0x42);
      put_byte(fp, 0x4D);
      /* DWORD bfSize */
      put_dword(fp, offset + bmsize * 4);
      /* UINT bfReserved1 */
      put_word(fp, 0);
      /* UNIT bfReserved2 */
      put_word(fp, 0);
      /* DWORD bfOffBits */
      put_dword(fp, offset);
      /* struct BMPINFOHEADER */
      /* DWORD biSize */
      put_dword(fp, 40);
      /* LONG biWidth */
      put_dword(fp, img->xsize);
      /* LONG biHeight */
      put_dword(fp, img->ysize);
      /* WORD biPlanes */
      put_word(fp, 1);
      /* WORD biBitCount */
      put_word(fp, bitcnt);
      /* DWORD biCompression */
      put_dword(fp, 0 /* BI_RGB (not compressed) */);
      /* DWORD biSizeImage */
      put_dword(fp, 0);
      /* LONG biXPelsPerMeter */
      put_dword(fp, img->xres);
      /* LONG biYPelsPerMeter */
      put_dword(fp, img->yres);
      /* DWORD biClrUsed */
      put_dword(fp, 0);
      /* DWORD biClrImportant */
      put_dword(fp, 0);
      /* struct RGBQUAD and pixel data bits */
      switch (img->type)
      {  case IMG_2: /* 2 colors (1 bit per pixel) */
            {  int k;
               for (k = 0; k < 2; k++)
               {  put_byte(fp, img->rgb[k].blue);
                  put_byte(fp, img->rgb[k].green);
                  put_byte(fp, img->rgb[k].red);
                  put_byte(fp, 0);
               }
               for (k = img->ysize - 1; k >= 0; k--)
               {  fwrite(img->scan + img->len * k, img->len, 1, fp);
                  fwrite("???", (- img->len) & 3, 1, fp);
               }
            }
            break;
         case IMG_4: /* 4 colors (2 bits per pixel) */
            {  int k, t;
               unsigned char *byte, val[2];
               for (k = 0; k < 16; k++)
               {  put_byte(fp, img->rgb[k & 3].blue);
                  put_byte(fp, img->rgb[k & 3].green);
                  put_byte(fp, img->rgb[k & 3].red);
                  put_byte(fp, 0);
               }
               for (k = img->ysize - 1; k >= 0; k--)
               {  byte = img->scan + img->len * k;
                  for (t = 0; t < img->len; t++, byte++)
                  {  val[0] = (unsigned char)
                        (((*byte >> 2) & 0x30) | ((*byte >> 4) & 0x03));
                     val[1] = (unsigned char)
                        (((*byte << 2) & 0x30) | ((*byte << 0) & 0x03));
                     put_byte(fp, val[0]);
                     put_byte(fp, val[1]);
                  }
                  fwrite("???", (- 2 * img->len) & 3, 1, fp);
               }
            }
            break;
         case IMG_16: /* 16 colors (4 bits per pixel) */
            {  int k;
               for (k = 0; k < 16; k++)
               {  put_byte(fp, img->rgb[k].blue);
                  put_byte(fp, img->rgb[k].green);
                  put_byte(fp, img->rgb[k].red);
                  put_byte(fp, 0);
               }
               for (k = img->ysize - 1; k >= 0; k--)
               {  fwrite(img->scan + img->len * k, img->len, 1, fp);
                  fwrite("???", (- img->len) & 3, 1, fp);
               }
            }
            break;
         case IMG_256: /* 256 colors (8 bits per pixel) */
            {  int k;
               for (k = 0; k < 256; k++)
               {  put_byte(fp, img->rgb[k].blue);
                  put_byte(fp, img->rgb[k].green);
                  put_byte(fp, img->rgb[k].red);
                  put_byte(fp, 0);
               }
               for (k = img->ysize - 1; k >= 0; k--)
               {  fwrite(img->scan + img->len * k, img->len, 1, fp);
                  fwrite("???", (- img->len) & 3, 1, fp);
               }
            }
            break;
         default:
            assert(img->type != img->type);
      }
      fflush(fp);
      if (ferror(fp))
      {  error("save_as_bmp: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
