/* glpmat/mat_vec.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- mat_vec - multiply matrix on vector (y := A * x)
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- double *mat_vec(double y[], MAT *A, double x[]);
--
-- *Description*
--
-- The mat_vec routine computes the product y := A * x, where A is a
-- sparse matrix of general kind, x and y are dense vectors. The matrix
-- A and the vector x remain unchanged. The input array x should contain
-- vector elements in x[1], x[2], ..., x[n], where n is the number of
-- columns of the matrix A. The output array y will contain vector
-- elements in y[1], y[2], ..., y[m], where m is the number of rows of
-- the matrix A. The arrays x and y should not overlap each other.
--
-- *Returns*
--
-- The mat_vec routine returns a pointer to the array y. */

double *mat_vec(double y[], MAT *A, double x[])
{     int i, j;
      for (i = 1; i <= A->m; i++) y[i] = 0.0;
      for (j = 1; j <= A->n; j++)
      {  double t = x[j];
         if (t != 0.0)
         {  ELEM *e;
            for (e = A->col[j]; e != NULL; e = e->col)
               y[e->i] += e->val * t;
         }
      }
      return y;
}

/* eof */
