/* glpprob/check_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpprob.h"

/*----------------------------------------------------------------------
-- check_lp - check LP data block for correctness.
--
-- *Synopsis*
--
-- #include "glpprob.h"
-- void check_lp(LP *lp);
--
-- *Description*
--
-- The check_lp routine checks the linear programming data block, which
-- lp points to, for correctness. In case of error the routine displays
-- an error message and terminates the program.
--
-- Note that the check_lp routine doesn't check the constraint matrix,
-- because the corresponding operation is extermely inefficient. It may
-- be checked additionally by the check_mat routine. */

void check_lp(LP *lp)
{     int k;
      if (lp->m < 1)
         fault("check_lp: invalid number of rows");
      if (lp->n < 1)
         fault("check_lp: invalid number of columns");
      for (k = 1; k <= lp->m+lp->n; k++)
      {  switch (lp->type[k])
         {  case 'F':
               if (!(lp->lb[k] == 0.0 && lp->ub[k] == 0.0))
err:              fault("check_lp: invalid bounds of row/column");
               break;
            case 'L':
               if (lp->ub[k] != 0.0) goto err;
               break;
            case 'U':
               if (lp->lb[k] != 0.0) goto err;
               break;
            case 'D':
               break;
            case 'S':
               if (lp->lb[k] != lp->ub[k]) goto err;
               break;
            default:
               fault("check_lp: invalid type of variable");
         }
      }
      if (!(lp->A->m == lp->m && lp->A->n == lp->n))
         fault("check_lp: invalid dimension of constraint matrix");
      if (!(lp->dir == '-' || lp->dir == '+'))
         fault("check_lp: invalid optimization direction flag");
      return;
}

/* eof */
