/* glpprob/transform_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glpprob.h"

/*----------------------------------------------------------------------
-- transform_lp - transform LP to the standard formulation.
--
-- *Synopsis*
--
-- #include "glpprob.h"
-- LP *transform_lp(LP *lp, int ref[]);
--
-- *Description*
--
-- The transform_lp routine transforms the LP problem, which lp points
-- to, to an equivalent LP problem, where all constraints are equalities
-- with non-negative right hand sides (i.e. all auxiliary variables are
-- fixed) and all structural variables are non-negative.
--
-- The array ref should have at least 1+m+n locations, where m and n are
-- respectively number of rows and columns of the *original* LP problem.
-- This array reflects the correspondence between rows and columns of
-- the original and the transformed problems and may be used to restore
-- the solution of the original problem using the known solution of the
-- transformed problem.
--
-- *Returns*
--
-- The transform_lp routine returns a pointer to LP data block, which
-- corresponds to the transformed problem. */

LP *transform_lp(LP *lp, int ref[])
{     LP *std;
      int m = lp->m, n = lp->n, mm, nn, ii, jj, k;
      /* determine number of rows (mm) and number of columns (nn) of
         the transformed LP */
      mm = nn = 0;
      for (k = 1; k <= m; k++)
      {  switch (lp->type[k])
         {  case 'F': break;
            case 'L': mm++; nn++; break;
            case 'U': mm++; nn++; break;
            case 'D': mm += 2; nn += 2; break;
            case 'S': mm++; break;
            default: assert(lp->type[k] != lp->type[k]);
         }
      }
      for (k = m+1; k <= m+n; k++)
      {  switch (lp->type[k])
         {  case 'F': nn += 2; break;
            case 'L': nn++; break;
            case 'U': nn++; break;
            case 'D': mm++; nn += 2; break;
            case 'S': break;
            default: assert(lp->type[k] != lp->type[k]);
         }
      }
      /* create the transformed LP (by default all auxiliary variable
         are fixed, all structural variable are non-negative) */
      if (mm == 0) mm = 1;
      if (nn == 0) nn = 1;
      std = create_lp(mm, nn);
      /* ii and jj are respectively number of current row and number of
         current column of the transformed LP */
      ii = jj = 0;
      /* transform rows (auxiliary variables) */
      for (k = 1; k <= m; k++)
      {  switch (lp->type[k])
         {  case 'F':
               /* source: -inf < (L.F.) < +inf */
               /* result: ignore free row */
               ref[k] = 0;
               break;
            case 'L':
               /* source: lb <= (L.F.) < +inf */
               /* result: (L.F.) - x' = lb, x' >= 0 */
               ii++; jj++;
               ref[k] = ii;
               new_elem(std->A, ii, jj, -1.0);
               std->lb[ii] = lp->lb[k];
               break;
            case 'U':
               /* source: -inf < (L.F.) <= ub */
               /* result: (L.F.) + x' = ub, x' >= 0 */
               ii++; jj++;
               ref[k] = ii;
               new_elem(std->A, ii, jj, +1.0);
               std->lb[ii] = lp->ub[k];
               break;
            case 'D':
               /* source: lb <= (L.F.) <= ub */
               /* result: (L.F.) + x' = ub, x' + x'' = ub - lb */
               ii++; jj++;
               ref[k] = ii;
               new_elem(std->A, ii, jj, -1.0);
               std->lb[ii] = lp->lb[k];
               ii++;
               new_elem(std->A, ii, jj, +1.0);
               jj++;
               new_elem(std->A, ii, jj, +1.0);
               std->lb[ii] = lp->ub[k] - lp->lb[k];
               break;
            case 'S':
               /* source: (L.F.) = lb */
               /* result: (L.F.) = lb */
               ii++;
               ref[k] = ii;
               std->lb[ii] = lp->lb[k];
               break;
            default:
               assert(lp->type[k] != lp->type[k]);
         }
      }
      /* transform columns (structural variables), build new constraint
         matrix and objective coefficients */
      for (k = m+1; k <= m+n; k++)
      {  ELEM *e;
         int j = k - m;
         switch (lp->type[k])
         {  case 'F':
               /* source: -inf < x < +inf */
               /* result: x = x' - x'', x' >= 0, x'' >= 0 */
               jj++;
               ref[k] = mm + jj;
               for (e = lp->A->col[j]; e != NULL; e = e->col)
               {  if (ref[e->i] != 0)
                     new_elem(std->A, ref[e->i], jj, +e->val);
               }
               std->c[jj] = +lp->c[j];
               jj++;
               for (e = lp->A->col[j]; e != NULL; e = e->col)
               {  if (ref[e->i] != 0)
                     new_elem(std->A, ref[e->i], jj, -e->val);
               }
               std->c[jj] = -lp->c[j];
               break;
            case 'L':
               /* source: lb <= x < +inf */
               /* result: x = lb + x', x' >= 0 */
               jj++;
               ref[k] = mm + jj;
               for (e = lp->A->col[j]; e != NULL; e = e->col)
               {  if (ref[e->i] != 0)
                  {  std->lb[ref[e->i]] -= e->val * lp->lb[k];
                     new_elem(std->A, ref[e->i], jj, e->val);
                  }
               }
               std->c[0] += lp->c[j] * lp->lb[k];
               std->c[jj] = lp->c[j];
               break;
            case 'U':
               /* source: -inf < x <= ub */
               /* result: x = ub - x', x' >= 0 */
               jj++;
               ref[k] = mm + jj;
               for (e = lp->A->col[j]; e != NULL; e = e->col)
               {  if (ref[e->i] != 0)
                  {  std->lb[ref[e->i]] += e->val * lp->ub[k];
                     new_elem(std->A, ref[e->i], jj, e->val);
                  }
               }
               std->c[0] -= lp->c[j] * lp->ub[k];
               std->c[jj] = lp->c[j];
               break;
            case 'D':
               /* source: lb <= x <= ub */
               /* result: x = lb + x', x' + x'' = ub - lb */
               jj++;
               ref[k] = mm + jj;
               for (e = lp->A->col[j]; e != NULL; e = e->col)
               {  if (ref[e->i] != 0)
                  {  std->lb[ref[e->i]] -= e->val * lp->lb[k];
                     new_elem(std->A, ref[e->i], jj, e->val);
                  }
               }
               std->c[0] += lp->c[j] * lp->lb[k];
               std->c[jj] = lp->c[j];
               ii++;
               new_elem(std->A, ii, jj, +1.0);
               jj++;
               new_elem(std->A, ii, jj, +1.0);
               std->lb[ii] = lp->ub[k] - lp->lb[k];
               break;
            case 'S':
               /* source: x = lb */
               /* result: just substitute */
               ref[k] = 0;
               for (e = lp->A->col[j]; e != NULL; e = e->col)
               {  if (ref[e->i] != 0)
                     std->lb[ref[e->i]] -= e->val * lp->lb[k];
               }
               std->c[0] += lp->c[j] * lp->lb[k];
               break;
            default:
               assert(lp->type[k] != lp->type[k]);
         }
      }
      /* end of transformation */
      assert(ii == mm && jj == nn);
      /* make all right hand sides (i.e. fixed values of auxiliary
         variables) of the transformed LP be non-negative */
      for (k = 1; k <= m; k++)
      {  ii = ref[k];
         if (ii != 0)
         {  assert(1 <= ii && ii <= mm);
            if (std->lb[ii] < 0.0)
            {  ELEM *e;
               ref[k] = - ref[k];
               std->lb[ii] = - std->lb[ii];
               for (e = std->A->row[ii]; e != NULL; e = e->row)
                  e->val = - e->val;
            }
         }
      }
      /* there may be rows x' + x'' = ub - lb, which also may have
         negative right hand sides (if lb > ub); formally we should
         transform them too */
      for (ii = 1; ii <= mm; ii++)
      {  if (std->lb[ii] < 0.0)
         {  ELEM *e;
            std->lb[ii] = - std->lb[ii];
            for (e = std->A->row[ii]; e != NULL; e = e->row)
               e->val = - e->val;
         }
      }
      /* lower and upper bounds of rows of the transformed LP should
         be identical */
      for (ii = 1; ii <= mm; ii++) std->ub[ii] = std->lb[ii];
      /* the transformed problem data block should be correct */
      check_lp(std);
      return std;
}

/* eof */
