/* glprsm/rsm_dual_opt.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include "glprsm.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- rsm_dual_opt - find optimal solution by dual simplex method.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_dual_opt(RSM *rsm);
--
-- *Description*
--
-- The rsm_dual_opt routine searches for optimal solution using dual
-- simplex method. It assumes that the current basis solution is dual
-- feasible.
--
-- This routine may be used to find primal feasible solution by means
-- of usual practice.
--
-- *Returns*
--
-- The rsm_dual_opt routine returns one of the following codes:
--
-- 0 - optimal solution found;
-- 1 - problem has no (primal) feasible solution;
-- 2 - numerical stability lost;
-- 3 - numerical problems with basis matrix. */

static RSM *_rsm;
/* revised simplex method common block */

static void display(int need)
{     /* display visual information which includes iteration number,
         value of objective function, sum of infeasibilities, and defect
         of basis solution (i.e. number of basis fixed variables); if
         the parameter need is not set, the routine displays information
         at least one second later after the last output */
      RSM *rsm = _rsm;
      if (need || rsm->t_last == 0 ||
          clock() - rsm->t_last > CLOCKS_PER_SEC)
      {  int def;
         double val, sum;
         val = rsm_objval(rsm);
         sum = rsm_infsum(rsm);
         def = rsm_defect(rsm);
         print(" %6d:   objval = %17.9e   infsum = %17.9e (%d)",
            rsm->iter, val, sum, def);
         rsm->t_last = clock();
      }
      return;
}

int rsm_dual_opt(RSM *rsm)
{     int m = rsm->m, ret;
      double *w;
      _rsm = rsm;
      /* allocate working array */
      if (rsm->dual_steep) w = ucalloc(1+m, sizeof(double));
      /* main loop starts here */
      for (;;)
      {  /* compute current values of basis variables */
         rsm_set_bbar(rsm);
         /* display information about current basis solution */
         display(0);
         /* compute simplex multipliers */
         rsm_set_pi(rsm);
         /* compute reduced costs of non-basis variables */
         rsm_set_cbar(rsm);
         /* check current solution for numerical stability */
         if (rsm_check_cbar(rsm, rsm->tol_dj))
         {  /* numerical stability lost */
            if (rsm->fin_out) display(1);
            ret = 2;
            break;
         }
         /* check current solution for primal feasibility */
         if (rsm_check_bbar(rsm, 0.30 * rsm->tol_bnd) == 0)
         {  /* the current basis is primal feasible */
            if (rsm->fin_out) display(1);
            ret = 0;
            break;
         }
         /* choose basis variable xB[p] */
         rsm_dual_row(rsm, 0.30 * rsm->tol_bnd);
         if (rsm->p == 0)
         {  /* it should never be because the current basis solution is
               primal infeasible */
            fault("rsm_dual_opt: internal logic error");
         }
         /* compute p-th row of the inverse inv(B) */
         rsm_set_bp(rsm);
         /* compute p-th row of the current simplex table */
         rsm_set_ap(rsm);
         /* choose non-basis variable xN[q] */
         if (!rsm->dual_relax)
         {  /* use standard "textbook" ratio test */
            rsm_dual_col(rsm, rsm->tol_piv);
         }
         else
         {  /* use technique proposed by P.Harris */
            rsm_harris_col(rsm, rsm->tol_piv, 0.10 * rsm->tol_dj);
         }
         if (rsm->q == 0)
         {  /* problem has no (primal) feasible solution */
            if (rsm->fin_out) display(1);
            ret = 1;
            break;
         }
         /* correct tagp if xB[p] is fixed variable (see comments to
            routines rsm_dual_row, rsm_dual_col, and rsm_harris col) */
         if (rsm->type[rsm->indb[rsm->p]] == LP_FX) rsm->tagp = LP_NS;
         /* update weights (if steepest edge pricing is used) */
         if (rsm->dual_steep) rsm_update_dzeta(rsm, w);
         /* compute q-th column of the current simplex table (this is
            not needed if steepest edge pricing is used, because this
            column is computed by the rsm_update_dzeta routine) */
         if (!rsm->dual_steep) rsm_set_aq(rsm);
         /* jump to the adjacent basis */
         if (rsm_change_basis(rsm) != 0)
         {  /* numerical problems with basis matrix */
            ret = 3;
            break;
         }
         /* check accuracy of updated weights (if required) */
         if (rsm->dual_steep && rsm->check_dzeta)
            print("check_dzeta: %g", rsm_check_dzeta(rsm));
         /* end of main loop */
      }
      /* free working array */
      if (rsm->dual_steep) ufree(w);
      /* return to the calling program */
      return ret;
}

/* eof */
