/* glprsm/rsm_exact_gamma.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glprsm.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- rsm_exact_gamma - compute exact weight (steepest edge technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double rsm_exact_gamma(RSM *rsm, int j);
--
-- *Description*
--
-- The rsm_exact_gamma routine computes exact value of the weight at
-- the non-basis variable xN[j] for the current basis using directly
-- the following formula:
--
--    gamma[j] = 1 + SUM alfa[i,j]^2, j = 1, ..., n,
--                    i
--
-- where alfa[i,j] is the element of the current simplex table placed
-- in i-th row and j-th column.
--
-- This operation is extremely inefficient, therefore it may be used
-- only for debugging purposes.
--
-- For details see:
-- D.Goldfarb, J.K.Reid. A practicable steepest-edge algorithm. Math.
-- Prog., 12, 1977, 361-71.
--
-- *Returns*
--
-- The rsm_exact_gamma routine returns the computed value of the weight
-- gamma[j]. */

double rsm_exact_gamma(RSM *rsm, int j)
{     ELEM *e;
      int m = rsm->m, n = rsm->n, k;
      double *aj, t;
      assert(1 <= j && j <= n);
      k = rsm->indn[j]; /* x[k] = xN[j] */
      /* aj = - inv(B) * N[j] */
      aj = ucalloc(1+m, sizeof(double));
      for (e = rsm->A->col[k]; e != NULL; e = e->col)
         aj[e->i] = e->val;
      rsm_ftran(rsm, aj, 0);
      t = 1.0;
      for (k = 1; k <= m; k++) t += aj[k] * aj[k];
      ufree(aj);
      return t;
}

/* eof */
