/* glprsm/rsm_find_feas.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include "glprsm.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- rsm_find_feas - find primal feasible solution.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_find_feas(RSM *rsm);
--
-- *Description*
--
-- The rsm_find_feas routine searches for primal feasible solution using
-- primal simplex method and implicit artificial variables technique.
--
-- At first the routine computes current values of the basis variables
-- and replaces basis variables, which violate their bounds, by implicit
-- artificial variables in order to construct feasible basis solution:
--
-- if bbar[i] < lB[i] - eps, i.e. if the basis variable xB[i] violates
-- its lower bound, the routine replaces the constraint xB[i] >= lB[i]
-- by constraint xB[i] <= lB[i] introducing implicit artificial variable
-- which satisfies its upper (!) bound;
--
-- if bbar[i] > ub[i] + eps, i.e. if the basis variable xB[i] violates
-- its upper bound, the routine replaces the constraint xB[i] <= uB[i]
-- by constraint xB[i] >= uB[i] introducing implicit artificial variable
-- which satisfies its lower (!) bound.
--
-- In both cases eps = tol*max(|lb[i]|,1) or eps = tol*max(|ub[i]|,1)
-- (depending on what bound is checked), where tol = 0.30*tol_bnd.
--
-- Should note that actually no new variables are introduced to the
-- problem. The routine just replaces types and values of bounds by new
-- ones.
--
-- Implicit artificial variables correspond to resduals. Therefore the
-- goal is to turn out all implicit variables from the basis that allows
-- to eliminate corresponding residuals. To do this the routine uses the
-- special objective function. If there is the implicit artificial
-- variable xB[i] >= lB[i], the routine sets its coefficient to +1 in
-- order to minimize the corresponding residual from the point of view
-- of the original problem. Analogously, if there is the implicit
-- artificial variable xB[i] <= uB[i], the routine sets its coefficient
-- to -1 again in order to minimize the corresponding residual. Other
-- coefficient are set to zero.
--
-- Should note that when some artificial variable becomes non-basis,
-- the corresponding residual vanishes. Hence the artificial objective
-- function is changed on each iteration of the simplex method.
--
-- Let the basis variable xB[p] leaves the basis. If it is the implicit
-- artificial variable (that recognized by changed type and bounds of
-- this variable), the routine restores its type and bounds to original
-- ones, because in the adjacent basis the corresponding residual will
-- be zero. 
--
-- In the case of degeneracy all implicit artificial variables may
-- become zero being basis variables. In this case the routine doesn't
-- wait these variables to leave the basis and stops the searching,
-- since such basis solution may be considered as feasible.
--
-- In any case (whether feasible solution was found or not) the routine
-- restores original types and bounds of all variables after the search
-- was finished.
--
-- *Returns*
--
-- The rsm_find_feas routine returns one of the following codes:
--
-- 0 - primal feasible solution found;
-- 1 - problem has no feasible solution;
-- 2 - numerical stability lost;
-- 3 - numerical problems with basis matrix. */

static RSM *_rsm;
/* revised simplex method common block */

static int *orig_type;
/* original types of variables */

static double *orig_lb, *orig_ub;
/* original bounds of variables */

static void display(int need)
{     /* display visual information which includes iteration number,
         value of objective function, sum of infeasibilities, and defect
         of basis solution (i.e. number of basis fixed variables); if
         the parameter need is not set, the routine displays information
         at least one second later after the last output */
      RSM *rsm = _rsm;
      if (need || rsm->t_last == 0 ||
          clock() - rsm->t_last > CLOCKS_PER_SEC)
      {  int *type, def;
         double *lb, *ub, val, sum;
         type = rsm->type, lb = rsm->lb, ub = rsm->ub;
         rsm->type = orig_type, rsm->lb = orig_lb, rsm->ub = orig_ub;
         val = rsm_objval(rsm);
         sum = rsm_infsum(rsm);
         def = rsm_defect(rsm);
         rsm->type = type, rsm->lb = lb, rsm->ub = ub;
         print(" %6d:   objval = %17.9e   infsum = %17.9e (%d)",
            rsm->iter, val, sum, def);
         rsm->t_last = clock();
      }
      return;
}

int rsm_find_feas(RSM *rsm)
{     int m = rsm->m, n = rsm->n, i, k, ret;
      double *y, *w, tol;
      _rsm = rsm;
      /* allocate working arrays */
      orig_type = ucalloc(1+m+n, sizeof(int));
      orig_lb = ucalloc(1+m+n, sizeof(double));
      orig_ub = ucalloc(1+m+n, sizeof(double));
      if (rsm->prim_steep)
      {  y = ucalloc(1+m, sizeof(double));
         w = ucalloc(1+m, sizeof(double));
      }
      /* save original types and bounds of variables */
      for (k = 1; k <= m+n; k++)
      {  orig_type[k] = rsm->type[k];
         orig_lb[k] = rsm->lb[k];
         orig_ub[k] = rsm->ub[k];
      }
      /* compute current values of basis variables */
      rsm_set_bbar(rsm);
      /* make artificial feasible basis solution */
      tol = 0.30 * rsm->tol_bnd;
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == LP_LO || rsm->type[k] == LP_DB ||
             rsm->type[k] == LP_FX)
         {  if (rsm_check_dev(rsm->bbar[i], rsm->lb[k], tol) == -2)
            {  /* xB[i] violates its lower bound */
               rsm->type[k] = LP_UP;
               rsm->lb[k] = 0.0;
               rsm->ub[k] = orig_lb[k];
               continue;
            }
         }
         if (rsm->type[k] == LP_UP || rsm->type[k] == LP_DB ||
             rsm->type[k] == LP_FX)
         {  if (rsm_check_dev(rsm->bbar[i], rsm->ub[k], tol) == +2)
            {  /* xB[i] violates its upper bound */
               rsm->type[k] = LP_LO;
               rsm->lb[k] = orig_ub[k];
               rsm->ub[k] = 0.0;
               continue;
            }
         }
      }
      /* main loop starts here */
      for (;;)
      {  /* compute current values of basis variables */
         rsm_set_bbar(rsm);
         /* display information about current basis solution */
         display(0);
         /* check current solution for numerical stability */
         if (rsm_check_bbar(rsm, rsm->tol_bnd))
         {  /* numerical stability lost */
            if (rsm->fin_out) display(1);
            ret = 2;
            break;
         }
         /* check current solution for primal feasibility */
         {  int *type; double *lb, *ub;
            type = rsm->type, lb = rsm->lb, ub = rsm->ub;
            rsm->type = orig_type, rsm->lb = orig_lb, rsm->ub = orig_ub;
            ret = rsm_check_bbar(rsm, 0.30 * rsm->tol_bnd);
            rsm->type = type, rsm->lb = lb, rsm->ub = ub;
            if (ret == 0)
            {  /* the current basis is primal feasible */
               if (rsm->fin_out) display(1);
               break;
            }
         }
         /* construct auxiliary objective function */
         for (k = 1; k <= m+n; k++)
         {  rsm->cost[k] = 0.0;
            if (rsm->type[k] == orig_type[k]) continue;
            /* if type[k] differs from orig_type[k], x[k] is implicit
               artificial variable (which should be basis variable!) */
            assert(rsm->posx[k] > 0);
            if (rsm->type[k] == LP_LO)
            {  /* x[k] should be decreased, since actually x[k] violates
                  its upper bound */
               rsm->cost[k] = +1.0;
            }
            else if (rsm->type[k] == LP_UP)
            {  /* x[k] should be increased, since actually x[k] violates
                  its lower bound */
               rsm->cost[k] = -1.0;
            }
            else
               assert(rsm->type[k] != rsm->type[k]);
         }
         /* compute simplex multipliers */
         rsm_set_pi(rsm);
         /* compute reduced costs of non-basis variables */
         rsm_set_cbar(rsm);
         /* choose non-basis variable xN[q] */
         rsm_pivot_col(rsm, rsm->tol_dj);
         if (rsm->q == 0)
         {  /* problem has no feasible solution */
            if (rsm->fin_out) display(1);
            ret = 1;
            break;
         }
         /* compute pivot column of simplex table */
         rsm_set_aq(rsm);
         /* choose basis variable xB[p] */
         if (!rsm->prim_relax)
         {  /* use standard "textbook" ratio test */
            rsm_pivot_row(rsm, rsm->tol_piv);
         }
         else
         {  /* use technique proposed by P.Harris */
            rsm_harris_row(rsm, rsm->tol_piv, 0.10 * rsm->tol_bnd);
         }
         if (rsm->p == 0)
         {  /* it should never be */
            fault("rsm_find_feas: internal logic error");
         }
         /* restore original bounds of xB[p] */
         if (rsm->p > 0)
         {  int p = rsm->p;
            k = rsm->indb[p]; /* x[k] = xB[p] */
            if (rsm->type[k] != orig_type[k])
            {  /* x[k] is implicit artificial variable */
               assert(rsm->tagp == LP_NL || rsm->tagp == LP_NU);
               rsm->tagp = (rsm->tagp == LP_NL ? LP_NU : LP_NL);
               rsm->type[k] = orig_type[k];
               rsm->lb[k]   = orig_lb[k];
               rsm->ub[k]   = orig_ub[k];
               if (rsm->type[k] == LP_FX) rsm->tagp = LP_NS;
            }
         }
         /* update weights (if steepest edge pricing is used) */
         if (rsm->prim_steep && rsm->p > 0) rsm_update_gamma(rsm, y, w);
         /* jump to the adjacent basis */
         if (rsm_change_basis(rsm) != 0)
         {  /* numerical problems with basis matrix */
            ret = 3;
            break;
         }
         /* check accuracy of updated weights (if required) */
         if (rsm->prim_steep && rsm->check_gamma)
            print("check_gamma: %g", rsm_check_gamma(rsm));
         /* end of main loop */
      }
      /* restore original bounds of variables */
      for (k = 1; k <= m+n; k++)
      {  rsm->type[k] = orig_type[k];
         rsm->lb[k] = orig_lb[k];
         rsm->ub[k] = orig_ub[k];
      }
      /* free working arrays */
      ufree(orig_type), ufree(orig_lb), ufree(orig_ub);
      if (rsm->prim_steep) ufree(y), ufree(w);
      /* return to the calling program */
      return ret;
}

/* eof */
