/* glprsm/rsm_set_aq.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_set_aq - compute pivot column of simplex table.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_set_aq(RSM *rsm);
--
-- *Description*
--
-- The rsm_set_aq routine computes the pivot column of the current
-- simplex table which corresponds to the chosen non-basis variable
-- xN[q], and stores this column to the array aq.
--
-- Since the computed column is q-th column of the matrix (-inv(B)*N),
-- the routine uses the following formula:
--
--    aq = - inv(B) * N[q],
--
-- where B is the current basis matrix, N[q] is the column of the
-- expanded constraint matrix A~, which corresponds to the variable
-- xN[q].
--
-- Should note that the variable xN[q] will enter the basis on the next
-- step and the column N[q] will replace other column of the current
-- basis matrix B. Therefore the routine rsm_set_aq tells the efi_tran
-- routine to save the transformed column (i.e. the column inv(B)*N[q])
-- through the parameter save. */

void rsm_set_aq(RSM *rsm)
{     ELEM *e;
      int m = rsm->m, n = rsm->n, i, k;
      double *z = rsm->aq;
      assert(1 <= rsm->q && rsm->q <= n);
      k = rsm->indn[rsm->q]; /* x[k] = xN[q] */
      /* z = N[q] */
      for (i = 1; i <= m; i++) z[i] = 0.0;
      for (e = rsm->A->col[k]; e != NULL; e = e->col)
         z[e->i] = + e->val;
      /* aq = - inv(B) * z */
      rsm_ftran(rsm, z, 1);
      for (i = 1; i <= m; i++) rsm->aq[i] = - z[i];
      return;
}

/* eof */
