/* glprsm/rsm_update_dzeta.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_update_dzeta - update weights (dual steepest edge).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_update_dzeta(RSM *rsm, double w[]);
--
-- *Description*
--
-- The rsm_update_dzeta routine recomputes weights, which are used in
-- the dual steepest edge technique. It is assumed that weights for the
-- current basis are stored on the array dzeta, p is the number of the
-- basis variable xB[p] chosen to leave the basis, and q is the number
-- of the non-basis variable xN[q] chosen to enter the basis. On exit
-- the array dzeta will contain weights for the adjacent basis.
--
-- The auxiliary array w should have at least m+1 elements, where m is
-- number of rows.
--
-- This routine should be called before the basis will be changed. */

void rsm_update_dzeta(RSM *rsm, double w[])
{     ELEM *e;
      int m = rsm->m, n = rsm->n, p = rsm->p, q = rsm->q, i, j, k;
      double *dzeta = rsm->dzeta, aiq, t1, t2;
      /* compute exact value of dzeta[p] for the current basis */
      dzeta[p] = 1.0;
      for (j = 1; j <= n; j++)
         dzeta[p] += rsm->ap[j] * rsm->ap[j];
      /* compute q-th column of the current simplex table */
      rsm_set_aq(rsm);
      /* compute auxiliary vector w */
      for (i = 1; i <= m; i++) w[i] = 0.0;
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j]; /* x[k] = xN[j] */
         if (rsm->ap[j] == 0.0) continue;
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            w[e->i] += rsm->ap[j] * e->val;
      }
      rsm_ftran(rsm, w, 0);
      /* recompute dzeta[i] for the adjacent basis (except dzeta[p]) */
      for (i = 1; i <= m; i++)
      {  if (i == p) continue;
         aiq = rsm->aq[i] / rsm->ap[q];
         t1 = dzeta[i] + (aiq * aiq) * dzeta[p] + 2 * aiq * w[i];
         t2 = 1 + aiq * aiq;
         dzeta[i] = t1 > t2 ? t1 : t2;
      }
      /* recompute dzeta[p] for the adjacent basis */
      dzeta[p] /= (rsm->ap[q] * rsm->ap[q]);
      return;
}

/* eof */
