/* glpset/scan_data.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "glpset.h"
#include "glpstr.h"

/*----------------------------------------------------------------------
-- scan_data - read data from data stream.
--
-- *Synopsis*
--
-- #include "glpset.h"
-- int scan_data(DATA *ds, ...);
--
-- *Description*
--
-- The scan_data reads data from data stream, which ds points to.
--
-- Data reading is controlled by variable parameter list. Each data
-- item, which has to be read, is represented by pair (fmt, ptr), where
-- fmt specifies the type of data item, ptr is a pointer to a location,
-- to which value of data item has to be stored. The end of parameter
-- list is marked by zero.
--
-- Currently the routine provides three types of data item, as shown in
-- the following table.
--
--    fmt   ptr         Description
--    ------------------------------------------------------
--    'i'   int *       Data item is a signed integer number
--    'e'   double *    Data item is a floating point number
--    's'   char *      Data item is a character string
--
-- For example, in order to read integer number ival, floating point
-- number fval, and character string str the following statement may be
-- used:
--
--    ... = scan_data(ds, 'i', &ival, 'e', &fval, 's', &str, 0);
--
-- All data items, which are read by one call to the scan_data routine,
-- should be placed in one text line. Space, HT and NL are considered as
-- delimiters and ignored; no other control characters are allowed. All
-- characters that follow after last data item to the end of line are
-- ignored. Character strings may be coded as in apostrophes as well as
-- without apostrophes. Each data item can contain up to 255 characters
-- (except optional apostrophes in case of character strings).
--
-- *Returns*
--
-- If the operation was successful, the scan_data routine returns zero.
-- Otherwise the routine prints an appropriate error message and returns
-- non-zero. */

int scan_data(DATA *ds, ...)
{     FILE *fp = ds->fp;
      va_list arg;
      int c, fmt, len;
      char str[255+1];
      va_start(arg, ds);
      /* get the first character of the current text line */
      ds->cn++;
      c = fgetc(fp);
      if (ferror(fp))
err1: {  error("%s:%d: reading error - %s", ds->fn, ds->cn,
            strerror(errno));
         goto fail;
      }
      if (feof(fp))
err2: {  error("%s:%d: unexpected end of file", ds->fn, ds->cn);
         goto fail;
      }
      /* scan and convert data items */
      for (;;)
      {  /* get type of the current data item */
         fmt = va_arg(arg, int);
         /* zero means the end of parameter list */
         if (fmt == 0) break;
         /* skip non-significant characters preceding the current data
            item */
         while (c == ' ' || c == '\t')
         {  c = fgetc(fp);
            if (ferror(fp)) goto err1;
            if (feof(fp)) goto err2;
         }
         /* check for premature end of line */
         if (c == '\n')
err3:    {  error("%s:%d: unexpected end of line", ds->fn, ds->cn);
            goto fail;
         }
         /* check for invalid control character */
         if (iscntrl(c))
err4:    {  error("%s:%d: invalid control character 0x%02X", ds->fn,
               ds->cn, c);
            goto fail;
         }
         /* scan the current data item */
         len = 0;
         if (!(fmt == 's' && c == '\''))
         {  /* ordinary data item */
            for (;;)
            {  if (c == ' ' || c == '\t' || c == '\n') break;
               if (iscntrl(c)) goto err4;
               if (len == 255)
               {  str[255] = '\0';
                  error("%s:%d: data item `%s...' too long", ds->fn,
                     ds->cn, str);
                  goto fail;
               }
               str[len++] = (char)c;
               c = fgetc(fp);
               if (ferror(fp)) goto err1;
               if (feof(fp)) goto err2;
            }
         }
         else
         {  /* apostrophized character string */
            int lev = 1;
            for (;;)
            {  c = fgetc(fp);
               if (ferror(fp)) goto err1;
               if (feof(fp)) goto err2;
               if (lev == 0)
               {  if (c == ' ' || c == '\t' || c == '\n') break;
                  if (iscntrl(c)) goto err4;
                  if (c != '\'')
                  {  error("%s:%d: character string is incorrect",
                        ds->fn, ds->cn);
                     goto fail;
                  }
                  lev = 1;
               }
               else
               {  if (c == '\n') goto err3;
                  if (iscntrl(c)) goto err4;
                  if (c == '\'')
                  {  lev = 0;
                     continue;
                  }
               }
               if (len == 255)
               {  error("%s:%d: character string too long", ds->fn,
                     ds->cn);
                  goto fail;
               }
               str[len++] = (char)c;
            }
         }
         str[len] = '\0';
         /* convert the current data item */
         switch (fmt)
         {  case 'i':
            {  int *ptr = va_arg(arg, int *);
               if (str2int(str, ptr) != 0)
               {  error("%s:%d: unable to convert `%s' to integer numbe"
                     "r", ds->fn, ds->cn, str);
                  goto fail;
               }
               break;
            }
            case 'e':
            {  double *ptr = va_arg(arg, double *);
               if (str2dbl(str, ptr) != 0)
               {  error("%s:%d: unable to convert `%s' to floating-poin"
                     "t number", ds->fn, ds->cn, str);
                  goto fail;
               }
               break;
            }
            case 's':
            {  char *ptr = va_arg(arg, char *);
               strcpy(ptr, str);
               break;
            }
            default:
               fault("scan_data: invalid type of data item");
         }
      }
      /* skip *all* characters following the last data item until end
         of line */
      while (c != '\n')
      {  c = fgetc(fp);
         if (ferror(fp)) goto err1;
         if (feof(fp)) goto err2;
         if (c == ' ' || c == '\t' || c == '\n') continue;
         if (iscntrl(c)) goto err4;
      }
      /* the current line has been processed */
      va_end(arg);
      return 0;
fail: va_end(arg);
      return 1;
}

/* eof */
