/* glpbbm/btrack_bestp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <stddef.h>
#include "glpbbm.h"

/*----------------------------------------------------------------------
-- btrack_bestp - select problem using the best projection heuristic.
--
-- *Synopsis*
--
-- #include "glpbbm.h"
-- BBNODE *btrack_bestp(BBDATA *bb);
--
-- *Description*
--
-- The btrack_bestp routine selects an appropriate problem which should
-- be considered next from the active list.
--
-- *Returns*
--
-- The btrack_bestp routine returns a pointer to the problem node.
--
-- *Heuristic*
--
-- If no integer feasible solution has been found, the btrack_bestp
-- routine selects the active problem whose parent problem has the best
-- value of the objective function, i.e. the tree is investigated in
-- the best-first-search manner.
--
-- If the best integer feasible solution is known, the routine selects
-- an active problem using the best projection heuristic. */

BBNODE *btrack_bestp(BBDATA *bb)
{     BBNODE *this = NULL, *node;
      if (!bb->found)
      {  /* no integer feasible solution has been found */
         double best = DBL_MAX;
         for (node = bb->first; node != NULL; node = node->right)
         {  if (best > node->up->objval)
               this = node, best = node->up->objval;
         }
      }
      else
      {  /* the best integer feasible solution is known */
         double best = DBL_MAX;
         for (node = bb->first; node != NULL; node = node->right)
         {  double deg, val;
            /* deg estimates degradation of the objective function per
               unit of the sum of integer infeasibilities */
            deg = (bb->best - bb->root->objval) / bb->root->infsum;
            /* val estimates optimal value of the objective function if
               the sum of integer infeasibilities would be zero */
            val = node->up->objval + deg * node->up->infsum;
            /* select the problem which have the best estimated optimal
               value of the objective function */
            if (best > val) this = node, best = val;
         }
      }
      return this;
}

/* eof */
