/* glpchol/create_adat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpchol.h"

/*----------------------------------------------------------------------
-- create_adat - create Cholesky factorization for S = A*D*A'.
--
-- *Synopsis*
--
-- #include "glpchol.h"
-- ADAT *create_adat(MAT *A);
--
-- *Description*
--
-- The create_adat routine creates Cholesky factorization for symmetric
-- positive definite matrix S = A*D*A', where A is rectangular matrix of
-- full rank, D is diagonal non-singular matrix. This operation include
-- only symbolic phase of factorization, therefore only pattern of A is
-- used, i.e. numeric values of elements of A as well as matrix D are
-- ignored.
--
-- Note that this routine doesn't include dense columns processing.
--
-- *Returns*
--
-- The create_adat routine returns a pointer to the data structure that
-- represents Cholesky factorization. */

ADAT *create_adat(MAT *A)
{     ADAT *adat;
      int m = A->m, n = A->n;
      adat = umalloc(sizeof(ADAT));
      adat->m = m;
      adat->n = n;
      adat->S = create_mat(m, m);
      aat_symb(adat->S, A, NULL);
      adat->chol = create_chol(adat->S);
      return adat;
}

/* eof */
