/* glpchol/min_deg.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpchol.h"
#include "glpqmd.h"

/*----------------------------------------------------------------------
-- min_deg - minimum degree ordering.
--
-- *Synopsis*
--
-- #include "glpchol.h"
-- void min_deg(MAT *A, PER *P);
--
-- *Description*
--
-- The min_deg routine uses minimum degree ordering in order to find
-- such permutation matrix P for the given sparse symmetric (positive
-- definite) matrix A, which minimizes number of non-zeros of the upper
-- triangular matrix U for Cholesky factorization P*A*P' = U'*U.
--
-- So far as the matrix A is symmetric, on entry it should contain only
-- upper triangular part. Only pattern of A is used, i.e. numeric values
-- of elements of the matrix A are ignored (they all are considered as
-- non-zeros). The matrix A is unchanged on exit.
--
-- Order of the resultant permutation matrix P should be equal to order
-- of the matrix A. On entry the initial contents of P is ignored.
--
-- *Method*
--
-- The min_deg routine is based on some subroutines from the SPARSPAK
-- package described in the book:
--
-- Alan George, Joseph W-H Liu. Computer Solution of Large Sparse
-- Positive Definite Systems. Prentice-Hall, 1981.
--
-- See the acknowledgements in comments to the routines genqmd, qmdrch,
-- qmdqt, qmdupd, and qmdmrg included into the partition GLPQMD. */

void min_deg(MAT *A, PER *P)
{     ELEM *e;
      int n = A->m, ne, ke, k;
      int *xadj, *adjncy, *deg, *marker, *rchset, *nbrhd, *qsize,
         *qlink, nofsub;
      if (!(A->n == n && P->n == n))
         fault("min_deg: inconsistent dimension");
      /* determine the length of the array adjncy */
      ne = 0;
      for (k = 1; k <= n; k++)
      {  for (e = A->row[k]; e != NULL; e = e->row)
         {  if (e->i > e->j)
               fault("min_deg: input matrix is not upper triangular");
            if (e->i != e->j) ne++;
         }
         for (e = A->col[k]; e != NULL; e = e->col)
            if (e->i != e->j) ne++;
      }
      /* allocate working storage */
      xadj = ucalloc(1+n+1, sizeof(int));
      adjncy = ucalloc(1+ne, sizeof(int));
      deg = ucalloc(1+n, sizeof(int));
      marker = ucalloc(1+n, sizeof(int));
      rchset = ucalloc(1+n, sizeof(int));
      nbrhd = ucalloc(1+n, sizeof(int));
      qsize = ucalloc(1+n, sizeof(int));
      qlink = ucalloc(1+n, sizeof(int));
      /* build adjacency structure (xadj, adjncy) */
      ke = 1;
      for (k = 1; k <= n; k++)
      {  xadj[k] = ke;
         for (e = A->row[k]; e != NULL; e = e->row)
            if (e->i != e->j) adjncy[ke] = e->j, ke++;
         for (e = A->col[k]; e != NULL; e = e->col)
            if (e->i != e->j) adjncy[ke] = e->i, ke++;
      }
      xadj[n+1] = ke;
      /* call the main minimimum degree ordering routine */
      genqmd(&n, xadj, adjncy, P->row, P->col, deg, marker, rchset,
         nbrhd, qsize, qlink, &nofsub);
      /* check the found permutation matrix P for correctness */
      check_per(P);
      /* free working storage */
      ufree(xadj);
      ufree(adjncy);
      ufree(deg);
      ufree(marker);
      ufree(rchset);
      ufree(nbrhd);
      ufree(qsize);
      ufree(qlink);
      /* return to the calling program */
      return;
}

/* eof */
