/* glpefi/build_efi.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpefi.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- build_efi - build EFI for given basis matrix.
--
-- *Synopsis*
--
-- #include "glpefi.h"
-- int build_efi(EFI *efi, MAT *A, int indb[]);
--
-- *Description*
--
-- The build_efi routine builds EFI for the given basis matrix B (see
-- below) in the form B = B0*H, where B0 = B and H = I (unity matrix).
-- This is used when: (a) the current EFI became inaccurate, (b) the
-- current EFI requires too much memory, and (c) the basis matrix B was
-- completely changed.
--
-- The given basis matrix B should be specified implicitly by the matrix
-- A and the array indb. The matrix A should have m rows, where m is the
-- order of the basis matrix B. The array indb should specify a list of
-- column numbers of the matrix A, which form the matrix B. These column
-- numbers should be placed in locations indb[1], indb[2], ..., indb[m].
--
-- *Returns*
--
-- If the EFI has been built, the routine returns zero. Otherwise, the
-- routine returns non-zero. The latter case can happen if the matrix B
-- is numerically singular or ill conditioned (in this case the partial
-- LU-factorization can be used to analyze what caused the problem; for
-- details see LU-factorization routines). */

#define maxtry 4

int build_efi(EFI *efi, MAT *A, int indb[])
{     static double tol[1+maxtry] = { 0.00, 0.01, 0.10, 0.40, 0.85 };
      int m = efi->m, ret, try;
      double Bmax, Ubig;
      if (A->m != m)
         fault("build_efi: invalid number of rows");
      /* try to build LU-factorization of the given basis matrix B */
      for (try = 1; try <= maxtry; try++)
      {  int i;
         /* U := B */
         clear_mat(efi->lu->U);
         for (i = 1; i <= m; i++)
         {  ELEM *e;
            int k = indb[i]; /* i-th column of B is k-th column of A */
            if (!(1 <= k && k <= A->n))
               fault("build_efi: invalid column list");
            for (e = A->col[k]; e != NULL; e = e->col)
            {  if (e->val != 0.0)
                  new_elem(efi->lu->U, e->i, i, e->val);
            }
         }
         /* build LU-factorization using gaussian elimination */
         ret = build_lu(efi->lu, efi->lu->U, tol[try], 1e+10, NULL,
            &Bmax, &Ubig);
         if (ret == 0 || try == maxtry) break;
      }
      /* reset eta-file (H := I) */
      reset_eta(efi->eta);
      /* clear transformed column flag */
      efi->flag = 0;
      return ret;
}

/* eof */
