/* glplp/extract_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"
#include "glplp.h"

/*----------------------------------------------------------------------
-- extract_lp - extract LP problem data from the workspace.
--
-- *Synopsis*
--
-- #include "glplp.h"
-- LP *extract_lp(void);
--
-- *Description*
--
-- The extract_lp routine extracts linear programming problem data from
-- the workspace (for details see GLPK API).
--
-- *Control parameters*
--
-- The behavior of the extract_lp routine depends on the following
-- control parameters:
--
-- obj_dir (optimization_flag);
-- obj_row (name of the objective function row);
-- sum_aij (multiplets summation flag);
-- tol_aij (tolerance for dropping tiny elements).
--
-- *Returns*
--
-- If the operation has been completed successfully, the extract_lp
-- routine returns a pointer to the LP data block. In case of error the
-- routine sends diagnostics to stderr and returns NULL. */

LP *extract_lp(void)
{     /* control parameters */
      int obj_dir, sum_aij;
      double tol_aij;
      char obj_row[GLP_MAX_NAME+1];
      /* local variables */
      LP *lp = NULL;
      int m, n, i, j, k, kase, ret;
      /* query relevant control parameters */
      glp_get_ipar("obj_dir", &obj_dir);
      glp_get_cpar("obj_row", obj_row);
      glp_get_ipar("sum_aij", &sum_aij);
      glp_get_rpar("tol_aij", &tol_aij);
      /* renumber and count rows */
      m = 0;
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW)) glp_set_seqn(GLP_ROW, ++m);
      if (m == 0)
      {  error("extract_lp: problem has no rows");
         goto fail;
      }
      /* renumber and count columns */
      n = 0;
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL)) glp_set_seqn(GLP_COL, ++n);
      if (n == 0)
      {  error("extract_lp: problem has no columns");
         goto fail;
      }
      /* create LP data block */
      lp = create_lp(m, n, 0);
      /* extract types and bounds of variables */
      for (kase = 0; kase <= 1; kase++)
      {  int what = (kase == 0 ? GLP_ROW : GLP_COL);
         for (ret = glp_first_item(what); ret == 0;
              ret = glp_next_item(what))
         {  int seqn, type;
            double lb, ub;
            glp_get_seqn(what, &seqn);
            glp_get_bounds(what, &type, &lb, &ub);
            k = (kase == 0 ? 0 : m) + seqn;
            insist(1 <= k && k <= m+n);
            switch (type)
            {  case GLP_FR:
                  lp->type[k] = 'F';
                  lp->lb[k] = lp->ub[k] = 0.0;
                  break;
               case GLP_LO:
                  lp->type[k] = 'L';
                  lp->lb[k] = lb, lp->ub[k] = 0.0;
                  break;
               case GLP_UP:
                  lp->type[k] = 'U';
                  lp->lb[k] = 0.0, lp->ub[k] = ub;
                  break;
               case GLP_DB:
                  lp->type[k] = 'D';
                  lp->lb[k] = lb, lp->ub[k] = ub;
                  insist(lb < ub);
                  break;
               case GLP_FX:
                  lp->type[k] = 'S';
                  lp->lb[k] = lp->ub[k] = lb;
                  break;
               default:
                  insist(type != type);
            }
         }
      }
      /* extract matrix of constraint coefficients */
      for (ret = glp_first_item(GLP_ROW); ret == 0;
           ret = glp_next_item(GLP_ROW))
      {  int ret;
         glp_get_seqn(GLP_ROW, &i);
         for (ret = glp_first_coef(GLP_ROW); ret == 0;
              ret = glp_next_coef(GLP_ROW))
         {  double val;
            glp_get_seqn(GLP_COL, &j);
            glp_get_coef(&val);
            new_elem(lp->A, i, j, val);
         }
      }
      /* preprocessing matrix of constraint coefficients */
      if (!sum_aij)
      {  /* the matrix should have no multiplets */
         if (check_mplets(lp->A))
         {  error("extract_lp: constraint matrix has multiplets");
            goto fail;
         }
         /* remove zero and tiny elements */
         scrape_mat(lp->A, tol_aij);
      }
      else
      {  /* replace all multiplets by their sums (includes removing
            zero and tiny elements) */
         sum_mplets(lp->A, tol_aij);
      }
      /* extract objective function */
      lp->dir = (obj_dir == GLP_MIN ? '-' : '+');
      for (j = 0; j <= n; j++) lp->c[j] = 0.0;
      if (obj_row[0] != '\0')
      {  ELEM *e;
         glp_find_item(GLP_ROW, obj_row);
         glp_get_seqn(GLP_ROW, &i);
         insist(1 <= i && i <= m);
         for (e = lp->A->row[i]; e != NULL; e = e->row)
            lp->c[e->j] = e->val;
      }
      /* extract kinds of columns */
      for (ret = glp_first_item(GLP_COL); ret == 0;
           ret = glp_next_item(GLP_COL))
      {  int kind;
         glp_get_seqn(GLP_COL, &j);
         glp_get_kind(GLP_COL, &kind);
         if (kind)
         {  if (lp->kind == NULL) lp->kind = ucalloc(1+n, sizeof(int));
            lp->kind[j] = 1;
         }
      }
      /* check LP data for correctness */
      check_lp(lp);
      /* no errors */
      return lp;
fail: /* error occured */
      if (lp != NULL) delete_lp(lp);
      return NULL;
}

/* eof */
