/* glpmat/mat_per.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- mat_per - permute matrix columns (A := A * P)
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *mat_per(MAT *A, PER *P, void *work[]);
--
-- *Description*
--
-- The mat_per routine computes the product A := A * P, where A is a
-- sparse matrix of general kind, P is a permutation matrix. The matrix
-- P remains unchanged. The auxiliary array work should have at least
-- 1+n elements, where n is the number of columns of the matrix A (if
-- this parameter is NULL, the routine automatically allocates and frees
-- the working array).
--
-- *Returns*
--
-- The mat_per returns a pointer to the matrix A. */

MAT *mat_per(MAT *A, PER *P, void *_work[])
{     int j;
      void **work = _work;
      if (A->n != P->n)
         fault("mat_per: product undefined");
      if (_work == NULL) work = ucalloc(1+A->n, sizeof(void *));
      for (j = 1; j <= A->n; j++) work[j] = A->col[j];
      for (j = 1; j <= A->n; j++)
      {  ELEM *e;
         A->col[j] = work[P->col[j]];
         for (e = A->col[j]; e != NULL; e = e->col) e->j = j;
      }
      if (_work == NULL) ufree(work);
      return A;
}

/* eof */
