/* glpmps/mps_to_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include <string.h>
#include "glpmps.h"

/*----------------------------------------------------------------------
-- mps_to_lp - convert linear programming problem from MPS to LP.
--
-- *Synopsis*
--
-- #include "glpmps.h"
-- LP *mps_to_lp(MPS *mps, char *obj, char *rhs, char *rng, char *bnd);
--
-- *Description*
--
-- The mps_to_lp routine converts the linear programming problem data
-- block from the object of MPS type (which mps points to) to the object
-- of LP type.
--
-- The character string obj specifies row name, which has to be used as
-- the objective function. If obj is empty string, the first row of N
-- type (free auxiliary variable) is used as the objective function (if
-- the problem has no rows of N type, it is assumed that the objective
-- function is identically equal to zero).
--
-- The character string rhs specifies the name of right-hand side (RHS)
-- vector. If rhs is empty string, the first RHS vector is used (if the
-- problem has at least one RHS vector).
--
-- The character string rng specifies the name of range vector. If rng
-- is empty string, the first range vector is used (if the problem has
-- at least one range vector).
--
-- The character string bnd specifies the name of bound vector. If bnd
-- is empty string, the first bound vector is used (if the problem has
-- at least one bound vector).
--
-- *Returns*
--
-- If the conversion was successful, the mps_to_lp routine returns a
-- pointer to an object of LP type, which represents linear programming
-- data block in standard format. In case of error the routine returns
-- NULL. */

LP *mps_to_lp(MPS *mps, char *obj, char *rhs, char *rng, char *bnd)
{     LP *lp;
      MPSROW *row; MPSCOL *col; MPSCQE *cqe; MPSBQE *bqe;
      int m = mps->n_row, n = mps->n_col, i, j, k, objrow, v;
      lp = create_lp(m, n, 1);
      /* process ROWS section */
      for (i = 1; i <= m; i++)
      {  k = i; /* x[k] = i-th auxiliary variable */
         row = mps->row[i];
         if (strcmp(row->type, "N") == 0)
            lp->type[k] = 'F';
         else if (strcmp(row->type, "G") == 0)
            lp->type[k] = 'L';
         else if (strcmp(row->type, "L") == 0)
            lp->type[k] = 'U';
         else if (strcmp(row->type, "E") == 0)
            lp->type[k] = 'S';
         else
         {  error("mps_to_lp: row `%s' has unknown type `%s'",
               row->name, row->type);
            goto fail;
         }
      }
      /* process COLUMNS section */
      for (j = 1; j <= n; j++)
      {  col = mps->col[j];
         /* add column elements to the constraint matrix */
         for (cqe = col->ptr; cqe != NULL; cqe = cqe->next)
            new_elem(lp->A, cqe->ind, j, cqe->val);
         /* set integer variable flag */
         lp->kind[j] = (col->flag ? 1 : 0);
      }
      /* determine row of the objective function */
      objrow = 0;
      for (i = 1; i <= m; i++)
      {  row = mps->row[i];
         if ((obj[0] == '\0' && strcmp(row->type, "N") == 0) ||
             (obj[0] != '\0' && strcmp(row->name, obj) == 0))
         {  objrow = i;
            break;
         }
      }
      if (obj[0] != '\0' && objrow == 0)
      {  error("mps_to_lp: objective function row `%s' not found",
            obj);
         goto fail;
      }
      /* copy coefficients of the objective function */
      if (objrow != 0)
      {  ELEM *e;
         for (e = lp->A->row[objrow]; e != NULL; e = e->row)
            lp->c[e->j] += e->val;
      }
      /* process RHS section */
      if (rhs[0] == '\0')
         v = (mps->n_rhs == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_rhs; v++)
            if (strcmp(mps->rhs[v]->name, rhs) == 0) break;
         if (v > mps->n_rhs)
         {  error("mps_to_lp: right-hand side vector `%s' not found",
               rhs);
            goto fail;
         }
      }
      if (v > 0)
      {  for (cqe = mps->rhs[v]->ptr; cqe != NULL; cqe = cqe->next)
         {  i = cqe->ind;
            k = i; /* x[k] = i-th auxiliary variable */
            row = mps->row[i];
            /* some MPS files contains RHS elements for rows of N type;
               these elements mean that if the corresponding auxiliary
               variable (i.e. row) is non-basis, it has the given RHS
               value (instead zero); however, these elements are ignored
               by this routine, except RHS element for the objective
               function row, which is treated as its constant term with
               opposite sign */
            if (strcmp(row->type, "N") == 0)
            {  if (i == objrow)
                  lp->c[0] = - cqe->val;
               else
                  /* nop */;
            }
            else if (strcmp(row->type, "G") == 0)
               lp->lb[k] = cqe->val;
            else if (strcmp(row->type, "L") == 0)
               lp->ub[k] = cqe->val;
            else if (strcmp(row->type, "E") == 0)
               lp->lb[k] = lp->ub[k] = cqe->val;
            else
               insist(row->type != row->type);
         }
      }
      /* process RANGES section */
      if (rng[0] == '\0')
         v = (mps->n_rng == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_rng; v++)
            if (strcmp(mps->rng[v]->name, rng) == 0) break;
         if (v > mps->n_rng)
         {  error("mps_to_lp: range vector `%s' not found", rng);
            goto fail;
         }
      }
      if (v > 0)
      {  for (cqe = mps->rng[v]->ptr; cqe != NULL; cqe = cqe->next)
         {  i = cqe->ind;
            k = i; /* x[k] = i-th auxiliary variable */
            row = mps->row[i];
            if (strcmp(row->type, "N") == 0)
            {  error("mps_to_lp: range vector entry refers to row `%s' "
                  " of N type", row->name);
               goto fail;
            }
            else if (strcmp(row->type, "G") == 0)
incr:          lp->ub[k] = lp->lb[k] + fabs(cqe->val);
            else if (strcmp(row->type, "L") == 0)
decr:          lp->lb[k] = lp->ub[k] - fabs(cqe->val);
            else if (strcmp(row->type, "E") == 0)
            {  if (cqe->val > 0.0) goto incr;
               if (cqe->val < 0.0) goto decr;
            }
            else
               insist(row->type != row->type);
            lp->type[k] = (lp->lb[k] == lp->ub[k] ? 'S' : 'D');
         }
      }
      /* process BOUNDS section */
      if (bnd[0] == '\0')
         v = (mps->n_bnd == 0 ? 0 : 1);
      else
      {  for (v = 1; v <= mps->n_bnd; v++)
            if (strcmp(mps->bnd[v]->name, bnd) == 0) break;
         if (v > mps->n_bnd)
         {  error("mps_to_lp: bound vector `%s' not found", bnd);
            goto fail;
         }
      }
      if (v > 0)
      {  for (k = m+1; k <= m+n; k++)
            lp->lb[k] = 0.0, lp->ub[k] = +DBL_MAX;
         for (bqe = mps->bnd[v]->ptr; bqe != NULL; bqe = bqe->next)
         {  j = bqe->ind;
            k = m + j; /* x[k] = j-th structural variable */
            col = mps->col[j];
            if (strcmp(bqe->type, "LO") == 0)
               lp->lb[k] = bqe->val;
            else if (strcmp(bqe->type, "UP") == 0)
               lp->ub[k] = bqe->val;
            else if (strcmp(bqe->type, "FX") == 0)
               lp->lb[k] = lp->ub[k] = bqe->val;
            else if (strcmp(bqe->type, "FR") == 0)
               lp->lb[k] = -DBL_MAX, lp->ub[k] = +DBL_MAX;
            else if (strcmp(bqe->type, "MI") == 0)
               lp->lb[k] = -DBL_MAX;
            else if (strcmp(bqe->type, "PL") == 0)
               lp->ub[k] = +DBL_MAX;
            else if (strcmp(bqe->type, "UI") == 0)
            {  /* integer structural variable with upper bound */
               lp->ub[k] = bqe->val;
               lp->kind[j] = 1;
            }
            else if (strcmp(bqe->type, "BV") == 0)
            {  /* binary structural variable */
               lp->lb[k] = 0.0, lp->ub[k] = 1.0;
               lp->kind[j] = 1;
            }
            else
            {  error("mps_to_lp: bound vector entry for column `%s' has"
                  " unknown type `%s'", col->name, bqe->type);
               goto fail;
            }
         }
         for (k = m+1; k <= m+n; k++)
         {  if (lp->lb[k] == -DBL_MAX && lp->ub[k] == +DBL_MAX)
               lp->type[k] = 'F', lp->lb[k] = lp->ub[k] = 0.0;
            else if (lp->ub[k] == +DBL_MAX)
               lp->type[k] = 'L', lp->ub[k] = 0.0;
            else if (lp->lb[k] == -DBL_MAX)
               lp->type[k] = 'U', lp->lb[k] = 0.0;
            else if (lp->lb[k] != lp->ub[k])
               lp->type[k] = 'D';
            else
               lp->type[k] = 'S';
         }
      }
      /* if the problem has no integer columns, free the array kind */
      for (j = 1; j <= n; j++) if (lp->kind[j]) break;
      if (j > n) ufree(lp->kind), lp->kind = NULL;
      /* return to the calling program */
      return lp;
fail: /* conversion failed */
      delete_lp(lp);
      return NULL;
}

/* eof */
