/* genqmd.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- THIS CODE IS THE RESULT OF TRANSLATION OF THE FORTRAN SUBROUTINE
-- GENQMD FROM THE BOOK:
--
-- ALAN GEORGE, JOSEPH W-H LIU. COMPUTER SOLUTION OF LARGE SPARSE
-- POSITIVE DEFINITE SYSTEMS. PRENTICE-HALL, 1981.
--
-- THE TRANSLATION HAS BEEN DONE WITH THE PERMISSION OF THE AUTHORS
-- OF THE ORIGINAL FORTRAN SUBROUTINE: ALAN GEORGE AND JOSEPH LIU,
-- UNIVERSITY OF WATERLOO, WATERLOO, ONTARIO, CANADA.
--
-- The translation was made by Andrew Makhorin <mao@mai2.rcnet.ru>,
-- <mao@gnu.org>, Department for Applied Informatics, Moscow Aviation
-- Institute, Moscow, Russia.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpqmd.h"

/*----------------------------------------------------------------------
-- genqmd - GENeral Quotient Minimum Degree algorithm.
--
-- *Synopsis*
--
-- #include "glpqmd.h"
-- void genqmd(int *neqns, int xadj[], int adjncy[], int perm[],
--    int invp[], int deg[], int marker[], int rchset[], int nbrhd[],
--    int qsize[], int qlink[], int *nofsub);
--
-- *Purpose*
--
-- This routine implements the minimum degree algorithm. It makes use
-- of the implicit representation of the elimination graph by quotient
-- graphs, and the notion of indistinguishable nodes.
--
-- *Caution*
--
-- The adjancy vector adjncy will be destroyed.
--
-- *Input parameters*
--
-- neqns  - number of equations;
-- (xadj, adjncy) -
--          the adjancy structure.
--
-- *Output parameters*
--
-- perm   - the minimum degree ordering;
-- invp   - the inverse of perm.
--
-- *Working parameters*
--
-- deg    - the degree vector. deg[i] is negative means node i has been
--          numbered;
-- marker - a marker vector, where marker[i] is negative means node i
--          has been merged with another nodeand thus can be ignored;
-- rchset - vector used for the reachable set;
-- nbrhd  - vector used for neighborhood set;
-- qsize  - vector used to store the size of indistinguishable
--          supernodes;
-- qlink  - vector used to store indistinguishable nodes, i, qlink[i],
--          qlink[qlink[i]], ... are the members of the supernode
--          represented by i.
--
-- *Program subroutines*
--
-- qmdrch, qmdqt, qmdupd.
----------------------------------------------------------------------*/

void genqmd(int *_neqns, int xadj[], int adjncy[], int perm[],
      int invp[], int deg[], int marker[], int rchset[], int nbrhd[],
      int qsize[], int qlink[], int *_nofsub)
{     int inode, ip, irch, j, mindeg, ndeg, nhdsze, node, np, num,
         nump1, nxnode, rchsze, search, thresh;
#     define neqns  (*_neqns)
#     define nofsub (*_nofsub)
      /* Initialize degree vector and other working variables. */
      mindeg = neqns;
      nofsub = 0;
      for (node = 1; node <= neqns; node++)
      {  perm[node] = node;
         invp[node] = node;
         marker[node] = 0;
         qsize[node] = 1;
         qlink[node] = 0;
         ndeg = xadj[node+1] - xadj[node];
         deg[node] = ndeg;
         if (ndeg < mindeg) mindeg = ndeg;
      }
      num = 0;
      /* Perform threshold search to get a node of min degree.
         Variable search point to where search should start. */
s200: search = 1;
      thresh = mindeg;
      mindeg = neqns;
s300: nump1 = num + 1;
      if (nump1 > search) search = nump1;
      for (j = search; j <= neqns; j++)
      {  node = perm[j];
         if (marker[node] >= 0)
         {  ndeg = deg[node];
            if (ndeg <= thresh) goto s500;
            if (ndeg < mindeg) mindeg = ndeg;
         }
      }
      goto s200;
      /* Node has minimum degree. Find its reachable sets by calling
         qmdrch. */
s500: search = j;
      nofsub += deg[node];
      marker[node] = 1;
      qmdrch(&node, xadj, adjncy, deg, marker, &rchsze, rchset, &nhdsze,
         nbrhd);
      /* Eliminate all nodes indistinguishable from node. They are given
         by node, qlink[node], ... . */
      nxnode = node;
s600: num++;
      np = invp[nxnode];
      ip = perm[num];
      perm[np] = ip;
      invp[ip] = np;
      perm[num] = nxnode;
      invp[nxnode] = num;
      deg[nxnode] = -1;
      nxnode = qlink[nxnode];
      if (nxnode > 0) goto s600;
      if (rchsze > 0)
      {  /* Update the degrees of the nodes in the reachable set and
            identify indistinguishable nodes. */
         qmdupd(xadj, adjncy, &rchsze, rchset, deg, qsize, qlink,
            marker, &rchset[rchsze+1], &nbrhd[nhdsze+1]);
         /* Reset marker value of nodes in reach set. Update threshold
            value for cyclic search. Also call qmdqt to form new
            quotient graph. */
         marker[node] = 0;
         for (irch = 1; irch <= rchsze; irch++)
         {  inode = rchset[irch];
            if (marker[inode] >= 0)
            {  marker[inode] = 0;
               ndeg = deg[inode];
               if (ndeg < mindeg) mindeg = ndeg;
               if (ndeg <= thresh)
               {  mindeg = thresh;
                  thresh = ndeg;
                  search = invp[inode];
               }
            }
         }
         if (nhdsze > 0)
            qmdqt(&node, xadj, adjncy, marker, &rchsze, rchset, nbrhd);
      }
      if (num < neqns) goto s300;
      return;
#     undef neqns
#     undef nofsub
}

/* eof */
