/* glprsm/eval_zeta.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glprsm.h"

/*----------------------------------------------------------------------
-- eval_zeta - compute row of the inverse.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void eval_zeta(RSM *rsm, int i, double zeta[]);
--
-- *Description*
--
-- The eval_zeta routine computes i-th row of the inverse, i.e. i-th
-- row of the matrix inv(B), where B is the current basis matrix, and
-- stores its elements into locations zeta[1], ..., zeta[m].
--
-- The i-th row of inv(B) is computed using the following formula:
--
--    zeta = inv(B') * e[i],
--
-- where B' is a matrix transposed to the current basis matrix B, e[i]
-- is the unity vector containing one in i-th position. */

void eval_zeta(RSM *rsm, int i, double zeta[])
{     int m = rsm->m, j;
      insist(1 <= i && i <= m);
      for (j = 1; j <= m; j++) zeta[j] = 0.0;
      zeta[i] = 1.0;
      btran(rsm, zeta);
      return;
}

/* eof */
