/* glpapi/glp_scale_prob.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpk.h"
#include "glpmat.h"

/*----------------------------------------------------------------------
-- glp_scale_prob - scale problem.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_scale_prob(LPI *lp, int how);
--
-- *Description*
--
-- The routine glp_scale_prob performs scaling the problem instance
-- depending on the parameter how:
--
-- 0 - equilibration scaling;
-- 1 - geometric mean scaling;
-- 2 - geometric mean scaling, then equilibration scaling.
--
-- In order to scale the problem the routine uses current coefficients
-- of the constrint matrix.
--
-- A result of this operation are row and column scale factors that are
-- stored by the routine into the problem object. */

void glp_scale_prob(LPI *lp, int how)
{     MAT *A;
      int m, n, i, j;
      double *R, *S;
      /* determine dimension of the constraint matrix */
      m = glp_get_num_rows(lp);
      if (m == 0) fault("glp_scale_prob: problem has no rows");
      n = glp_get_num_cols(lp);
      if (n == 0) fault("glp_scale_prob: problem has no columns");
      /* obtain the constraint matrix */
      A = create_mat(m, n);
      {  int *cn = ucalloc(1+n, sizeof(int));
         double *ai = ucalloc(1+n, sizeof(double));
         for (i = 1; i <= m; i++)
         {  int nz = glp_get_row_coef(lp, i, cn, ai);
            for (j = 1; j <= nz; j++)
               new_elem(A, i, cn[j], ai[j]);
         }
         ufree(cn);
         ufree(ai);
      }
      /* compute scaling matrices */
      R = ucalloc(1+m, sizeof(double));
      S = ucalloc(1+n, sizeof(double));
      for (i = 1; i <= m; i++) R[i] = 1.0;
      for (j = 1; j <= n; j++) S[j] = 1.0;
      switch (how)
      {  case 0:
            /* equilibration scaling */
            eq_scaling(A, R, S, 0);
            break;
         case 1:
            /* geometric mean scaling */
            gm_scaling(A, R, S, 0, 0.01, 20);
            break;
         case 2:
            /* geometric mean scaling, then equilibration scaling */
            gm_scaling(A, R, S, 0, 0.01, 20);
            eq_scaling(A, R, S, 0);
            break;
         default:
            fault("glp_scale_prob: how = %d; invalid parameter", how);
      }
      /* store scale factors into the problem object */
      for (i = 1; i <= m; i++) glp_set_row_fctr(lp, i, R[i]);
      for (j = 1; j <= n; j++) glp_set_col_fctr(lp, j, S[j]);
      /* free working arrays and return */
      delete_mat(A);
      ufree(R);
      ufree(S);
      return;
}

/* eof */
