/* glpchol/create_chol.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpchol.h"

/*----------------------------------------------------------------------
-- create_chol - create Cholesky factorization.
--
-- *Synopsis*
--
-- #include "glpchol.h"
-- CHOL *create_chol(MAT *A);
--
-- *Description*
--
-- The create_chol routine creates Cholesky factorization for the given
-- symmetric positive definite matrix A. This operation includes only
-- symbolic phase of factorization, therefore only pattern of A is used,
-- i.e. numeric values of elements of A are ignored (all its elements
-- are considered as non-zeros).
--
-- *Returns*
--
-- The create_chol routine returns a pointer to the data structure that
-- represents Cholesky factorization. */

CHOL *create_chol(MAT *A)
{     CHOL *chol;
      int n = A->m;
      if (A->m != A->n)
         fault("create_chol: invalid matrix");
      chol = umalloc(sizeof(CHOL));
      chol->n = n;
      chol->P = create_per(n);
      chol->U = create_mat(n, n);
      chol->sing = -1;
      /* determine permutation matrix P, which minimizes number of
         non-zeros in the matrix U */
      min_deg(A, chol->P);
      /* compute permuted matrix A~ = P*A*P' */
      per_sym(chol->P, A, NULL);
      /* compute symbolic Cholesky factorization A~ = U'*U */
      chol_symb(A, chol->U, NULL, NULL, NULL);
      /* restore original matrix A = P'*A~*P */
      inv_per(chol->P);
      per_sym(chol->P, A, NULL);
      inv_per(chol->P);
      return chol;
}

/* eof */
