/* glprsm/check_cbar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glprsm.h"

/*----------------------------------------------------------------------
-- check_cbar - check basis solution for dual feasibility.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int check_cbar(RSM *rsm, double c[], double cbar[], double tol);
--
-- *Description*
--
-- The check_cbar routine checks the given basis solution for dual
-- feasibility.
--
-- The array c should contain the expanded vector c of coefficients of
-- the objective function in locations c[1], ..., c[m+n].
--
-- The array cbar should contain the given reduced costs of non-basic
-- variables d = (d_1, ..., d_n) in locations cbar[1], ..., cbar[n]
-- respectively. It is assumed that the reduced costs were computed
-- using the vector c passed to this routine.
--
-- The parameter tol > 0 is a relative tolerance.
--
-- In order to see of the given basis solution is dual feasible the
-- routine checks the dual feasibility conditions which using the
-- relative tolerance are the following:
--
--    if (xN)j in xF then -eps <= dj <= +eps
--    if (xN)j in xL then dj >= +eps
--    if (xN)j in xU then dj <= -eps
--    if (xN)j in xS then -inf < dj < +inf
--
-- where
--
--    eps = tol * max(1, |(cN)j|)
--
-- (the absolute tolerance eps reflects that the reduced cost on
-- non-basic variable (xN)j is the difference dj = (cN)j - pi'*Nj).
--
-- *Returns*
--
-- The the given basic solution is dual feasible (i.e. it satisfies to
-- the conditions above), the check_cbar routine returns zero. Otherwise
-- the routine returns non-zero. */

int check_cbar(RSM *rsm, double c[], double cbar[], double tol)
{     int n = rsm->n, j, k;
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j]; /* x[k] = xN[j] */
         if (rsm->tagn[j] == 'F' || rsm->tagn[j] == 'L')
         {  /* xN[j] can increase */
            if (check_rr(cbar[j] + c[k], c[k], tol) < -1) return 1;
         }
         if (rsm->tagn[j] == 'F' || rsm->tagn[j] == 'U')
         {  /* xN[j] can decrease */
            if (check_rr(cbar[j] + c[k], c[k], tol) > +1) return 1;
         }
      }
      return 0;
}

/* eof */
