/* glpapi/glp_delete_cols.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_delete_cols - delete columns from problem instance.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_delete_cols(LPI *lp, int flag[]);
--
-- *Description*
--
-- The routine glp_delete_cols deletes specified subset of columns from
-- the problem object.
--
-- The array flag should have at least 1+n locations, where n is number
-- of columns in the problem object. The location flag[0] is not used.
-- If flag[j] (1 <= j <= n) is non-zero, the routine deletes the j-th
-- column. Otherwise, if flag[j] is zero, the j-th column is kept.
--
-- *Complexity*
--
-- The time complexity is O(nz), where nz is total number of elements
-- in the constraint matrix before operation. */

void glp_delete_cols(LPI *lp, int flag[])
{     int j, n;
      glp_clear_mat_cols(lp, flag);
      n = 0;
      for (j = 1; j <= lp->n; j++)
      {  if (flag[j])
         {  glp_set_col_name(lp, j, NULL);
            free_atom(lp->col_pool, lp->col[j]);
         }
         else
         {  ELEM *e;
            n++;
            lp->col[n] = lp->col[j];
            lp->col[n]->seqn = n;
            for (e = lp->col[n]->ptr; e != NULL; e = e->col) e->j = n;
         }
      }
      lp->n = n;
      return;
}

/* eof */
