/* glpapi/glp_get_col_soln.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_get_col_soln - obtain column solution information.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_get_col_soln(LPI *lp, int j, int *tagx, double *valx,
--    double *dx);
--
-- *Description*
--
-- The routine glp_get_col_soln stores status, primal value, and dual
-- value of the j-th column (structural variable) to locations, which
-- the parameters tagx, valx, and dx point to, respectively.
--
-- If some of the pointers tagx, valx, or dx is NULL, the corresponding
-- value is not stored.
--
-- The status code has the following meaning:
--
-- 'B' - basic variable;
-- 'L' - non-basic variable on its lower bound;
-- 'U' - non-basic variable on its upper bound;
-- 'F' - non-basic free (unbounded) variable;
-- 'S' - non-basic fixed variable.
--
-- Should note that if the routine glp_get_bstat reports that the basis
-- is undefined, the status code is not applicable. Analogously, if the
-- routine glp_get_status reports that the solution is undefined, primal
-- and dual values are meaningless. */

void glp_get_col_soln(LPI *lp, int j, int *tagx, double *valx,
      double *dx)
{     if (!(1 <= j && j <= lp->n))
         fault("glp_get_col_soln: j = %d; invalid column number", j);
      if (tagx != NULL) *tagx = lp->col[j]->tagx;
      if (valx != NULL) *valx = lp->col[j]->valx;
      if (dx != NULL) *dx = lp->col[j]->dx;
      return;
}

/* eof */
