/* glpapi/glp_write_mps.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "glpk.h"
#include "glpavl.h"

/*----------------------------------------------------------------------
-- glp_init_wmps - initialize parameter block by default values.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_init_wmps(struct wmps *parm);
--
-- *Description*
--
-- The routine glp_init_wmps() initializes parameter block passed to the
-- routine glp_write_mps() by default values. */

void glp_init_wmps(struct wmps *parm)
{     parm->prob_info = 1;
      parm->make_obj = 2;
      parm->use_names = 1;
      parm->one_entry = 0;
      parm->pedantic = 0;
      parm->skip_empty = 0;
      return;
}

/*----------------------------------------------------------------------
-- glp_write_mps - write problem data using MPS format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_write_mps(LPI *lp, char *fname, struct wmps *parm);
--
-- *Description*
--
-- The routine glp_write_mps() writes LP problem data using MPS format
-- to the output text file whose name is the character string parm.
--
-- The parameter parm is a pointer to the parameter block used by the
-- routine. On entry this block should be initialized using the routine
-- glp_init_wmps(). It is allowed to specify NULL, in which case default
-- values are used.
--
-- Description of MPS format can be found in the document "GLPK User's
-- Guide".
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - invalid control parameters;
-- 2 - operation failed due to errors.
--
-- In case of non-zero return code the routine sends all diagnostics to
-- stderr. */

typedef char mps_name[8+1];
/* standard MPS names (contain 1 up to 8 characters) */

static char *plain_name(int what, int ij);
/* generate plain row/column name */

static void make_names(LPI *lp, int what, mps_name alias[]);
/* generate standard MPS names using original names */

static char *mps_number(double val);
/* convert number to standard 12-character MPS format */

int glp_write_mps(LPI *lp, char *fname, struct wmps *parm)
{     struct wmps _parm;
      FILE *fp;
      mps_name *row_name = NULL, *col_name = NULL;
      int marker = 0; /* intorg/intend marker count */
      int nrows, ncols, make_obj, i, j, flag, *rn;
      double *aj;
      /* if parameter block is not specified, create the dummy one */
      if (parm == NULL)
      {  parm = &_parm;
         glp_init_wmps(parm);
      }
      /* check control parameters for correctness */
      if (!(parm->prob_info == 0 || parm->prob_info == 1))
      {  error("glp_write_mps: prob_info = %d; invalid parameter",
            parm->prob_info);
         return 1;
      }
      if (!(0 <= parm->make_obj && parm->make_obj <= 2))
      {  error("glp_write_mps: make_obj = %d; invalid parameter",
            parm->make_obj);
         return 1;
      }
      if (!(parm->use_names == 0 || parm->use_names == 1))
      {  error("glp_write_mps: use_names = %d; invalid parameter",
            parm->use_names);
         return 1;
      }
      if (!(parm->one_entry == 0 || parm->one_entry == 1))
      {  error("glp_write_mps: one_entry = %d; invalid parameter",
            parm->one_entry);
         return 1;
      }
      if (!(parm->pedantic == 0 || parm->pedantic == 1))
      {  error("glp_write_mps: pedantic = %d; invalid parameter",
            parm->pedantic);
         return 1;
      }
      if (!(parm->skip_empty == 0 || parm->skip_empty == 1))
      {  error("glp_write_mps: prob_info = %d; invalid parameter",
            parm->skip_empty);
         return 1;
      }
      /* control parameters are correct */
      print("glp_write_mps: writing problem data to `%s'...", fname);
      /* open the output text file */
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("glp_write_mps: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* determine number of rows and number of columns */
      nrows = glp_get_num_rows(lp);
      ncols = glp_get_num_cols(lp);
      /* the problem should contain at least one row and one column */
      if (nrows == 0)
      {  error("glp_write_mps: problem has no rows");
         goto fail;
      }
      if (ncols == 0)
      {  error("glp_write_mps: problem has no columns");
         goto fail;
      }
      /* determine whether the routine should output objective function
         row */
      make_obj = parm->make_obj;
      if (make_obj == 2)
      {  for (i = 1; i <= nrows; i++)
         {  int type;
            glp_get_row_bnds(lp, i, &type, NULL, NULL);
            if (type == 'F')
            {  make_obj = 0;
               break;
            }
         }
      }
      /* allocate arrays for 8-character row and column names */
      row_name = ucalloc(1+nrows, sizeof(mps_name));
      col_name = ucalloc(1+ncols, sizeof(mps_name));
      /* generate 8-character name for the objective function row */
      strcpy(row_name[0], plain_name('R', 0));
      /* generate 8-character names for rows and columns */
      if (parm->use_names)
      {  /* use original row and column names as templates */
         make_names(lp, 'R', row_name);
         make_names(lp, 'C', col_name);
      }
      else
      {  /* generate plain names based on sequential numbers */
         for (i = 1; i <= nrows; i++)
            strcpy(row_name[i], plain_name('R', i));
         for (j = 1; j <= ncols; j++)
            strcpy(col_name[j], plain_name('C', j));
      }
      /* write comments cards (if required) */
      if (parm->prob_info)
      {  char *name = glp_get_prob_name(lp);
         int ni = glp_get_num_int(lp);
         int nb = glp_get_num_bin(lp);
         int nz = glp_get_num_nz(lp);
         if (name == NULL) name = "UNKNOWN";
         fprintf(fp, "* Problem:    %.31s\n", name);
         fprintf(fp, "* Rows:       %d\n", nrows);
         if (glp_get_num_int(lp) == 0)
         fprintf(fp, "* Columns:    %d\n", ncols);
         else
         fprintf(fp, "* Columns:    %d (%d integer, %d binary)\n",
            ncols, ni, nb);
         if (make_obj)
         {  for (j = 1; j <= ncols; j++)
               if (glp_get_obj_coef(lp, j) != 0.0) nz++;
         }
         fprintf(fp, "* Non-zeros:  %d\n", nz);
         fprintf(fp, "*\n");
      }
      /* write NAME indicator card */
      {  char *name = glp_get_prob_name(lp);
         if (name == NULL)
            fprintf(fp, "NAME");
         else
            fprintf(fp, "NAME          %.8s\n", name);
      }
      /* write ROWS section */
      fprintf(fp, "ROWS\n");
      if (make_obj)
         fprintf(fp, " %c  %s\n", 'N', row_name[0]);
      for (i = 1; i <= nrows; i++)
      {  int type;
         glp_get_row_bnds(lp, i, &type, NULL, NULL);
         switch (type)
         {  case 'F': type = 'N'; break;
            case 'L': type = 'G'; break;
            case 'U': type = 'L'; break;
            case 'D': type = 'E'; break;
            case 'S': type = 'E'; break;
            default: insist(type != type);
         }
         fprintf(fp, " %c  %s\n", type, row_name[i]);
      }
      /* write COLUMNS section */
      fprintf(fp, "COLUMNS\n");
      rn = ucalloc(1+nrows, sizeof(int));
      aj = ucalloc(1+nrows, sizeof(double));
      for (j = 1; j <= ncols; j++)
      {  int nl = 1, kind, nz, t;
         char *name;
         name = col_name[j];
         kind = glp_get_col_kind(lp, j);
         insist(kind == 'C' || kind == 'I');
         if (kind == 'I' && marker % 2 == 0)
         {  /* open new intorg/intend group */
            marker++;
            fprintf(fp, "    MARK%04d  'MARKER'                 'INTORG"
               "'\n" , marker);
         }
         else if (kind == 'C' && marker % 2 == 1)
         {  /* close the current intorg/intend group */
            marker++;
            fprintf(fp, "    MARK%04d  'MARKER'                 'INTEND"
               "'\n" , marker);
         }
         /* obtain j-th column of the constraint matrix */
         nz = glp_get_col_coef(lp, j, rn, aj);
         rn[0] = 0;
         aj[0] = (make_obj ? glp_get_obj_coef(lp, j) : 0.0);
         if (nz == 0 && aj[0] == 0.0 && !parm->skip_empty)
            fprintf(fp, "    %-8s  %-8s  %12s   $ empty column\n",
               name, row_name[1], mps_number(0.0));
         for (t = aj[0] != 0.0 ? 0 : 1; t <= nz; t++)
         {  if (nl)
               fprintf(fp, "    %-8s  ", name);
            else
               fprintf(fp, "   ");
            fprintf(fp, "%-8s  %12s",
               row_name[rn[t]], mps_number(aj[t]));
            if (!parm->one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
            if (!parm->pedantic) name = "";
         }
         if (!nl) fprintf(fp, "\n");
      }
      if (marker % 2 == 1)
      {  /* close the last intorg/intend group (if not closed) */
         marker++;
         fprintf(fp, "    M%07d  'MARKER'                 'INTEND'\n",
            marker);
      }
      ufree(rn);
      ufree(aj);
      /* write RHS section */
      flag = 0;
      {  int nl = 1;
         char *name = (parm->pedantic ? "RHS1" : "");
         for (i = make_obj ? 0 : 1; i <= nrows; i++)
         {  int type;
            double lb, ub, rhs;
            if (i == 0)
               type = 'F', lb = ub = 0.0;
            else
               glp_get_row_bnds(lp, i, &type, &lb, &ub);
            switch (type)
            {  case 'F':
                  /* if the current row is the objective function row,
                     right-hand side is set to the constant term of the
                     objective function with opposite sign; in other
                     cases right-hand side is not used */
                  rhs = (i == 0 ? - glp_get_obj_coef(lp, 0) : 0.0);
                  break;
               case 'L':
                  rhs = lb; break;
               case 'U':
                  rhs = ub; break;
               case 'D':
                  rhs = (ub > 0.0 ? lb : ub); break;
               case 'S':
                  rhs = lb; break;
               default:
                  insist(type != type);
            }
            if (rhs == 0.0) continue;
            if (!flag) fprintf(fp, "RHS\n"), flag = 1;
            if (nl)
                fprintf(fp, "    %-8s  ", name);
            else
                fprintf(fp, "   ");
            fprintf(fp, "%-8s  %12s", row_name[i], mps_number(rhs));
            if (!parm->one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
         }
         if (!nl) fprintf(fp, "\n");
      }
      /* write RANGES section */
      flag = 0;
      {  int nl = 1;
         char *name = (parm->pedantic ? "RNG1" : "");
         for (i = 1; i <= nrows; i++)
         {  int type;
            double lb, ub, rng;
            glp_get_row_bnds(lp, i, &type, &lb, &ub);
            if (type != 'D') continue;
            if (!flag) fprintf(fp, "RANGES\n"), flag = 1;
            if (nl)
                fprintf(fp, "    %-8s  ", name);
            else
                fprintf(fp, "   ");
            rng = (ub > 0.0 ? ub - lb : lb - ub);
            fprintf(fp, "%-8s  %12s", row_name[i], mps_number(rng));
            if (!parm->one_entry) nl = 1 - nl;
            if (nl) fprintf(fp, "\n");
         }
         if (!nl) fprintf(fp, "\n");
      }
      /* write BOUNDS section */
      flag = 0;
      {  char *name = (parm->pedantic ? "BND1" : "");
         for (j = 1; j <= ncols; j++)
         {  int type;
            double lb, ub;
            glp_get_col_bnds(lp, j, &type, &lb, &ub);
            if (type == 'L' && lb == 0.0) continue;
            if (!flag) fprintf(fp, "BOUNDS\n"), flag = 1;
            switch (type)
            {  case 'F':
                  fprintf(fp, " FR %-8s  %-8s\n", name, col_name[j]);
                  break;
               case 'L':
                  fprintf(fp, " LO %-8s  %-8s  %12s\n", name,
                     col_name[j], mps_number(lb));
                  break;
               case 'U':
                  fprintf(fp, " MI %-8s  %-8s\n", name, col_name[j]);
                  fprintf(fp, " UP %-8s  %-8s  %12s\n", name,
                     col_name[j], mps_number(ub));
                  break;
               case 'D':
                  if (lb != 0.0)
                  fprintf(fp, " LO %-8s  %-8s  %12s\n", name,
                     col_name[j], mps_number(lb));
                  fprintf(fp, " UP %-8s  %-8s  %12s\n", name,
                     col_name[j], mps_number(ub));
                  break;
               case 'S':
                  fprintf(fp, " FX %-8s  %-8s  %12s\n", name,
                     col_name[j], mps_number(lb));
                  break;
               default:
                  insist(type != type);
            }
         }
      }
      /* write ENDATA indicator card */
      fprintf(fp, "ENDATA\n");
      /* free working arrays */
      ufree(row_name), row_name = NULL;
      ufree(col_name), col_name = NULL;
      /* close the output text file */
      fflush(fp);
      if (ferror(fp))
      {  error("glp_write_mps: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      /* returns to the calling program */
      return 0;
fail: /* the operation failed */
      if (row_name != NULL) ufree(row_name);
      if (col_name != NULL) ufree(col_name);
      if (fp != NULL) fclose(fp);
      return 2;
}

/*----------------------------------------------------------------------
-- plain_name - generate plain row/column name.
--
-- This routine returns a pointer to a static buffer that contains name
-- of i-th row (if what = 'R') or of j-th column (if what == 'C'). */

static char *plain_name(int what, int ij)
{     static mps_name name;
      char *t;
      sprintf(name, "%c%7d", what, ij);
      for (t = name; *t; t++) if (*t == ' ') *t = '_';
      return name;
}

/*----------------------------------------------------------------------
-- make_names - generate standard MPS names using original names.
--
-- This routine tries to make names of rows (if what = 'R') or columns
-- (if what = 'C'), whose length doesn't exceed 8 chars, using original
-- row and column names as templates. The result names are placed in
-- alias[1], ..., alias[n], where n is the number of rows/columns. */

static void make_names(LPI *lp, int what, mps_name alias[])
{     AVLTREE *tree;
      int mn, ij;
      tree = create_avl((int (*)(void *, void *))strcmp);
      switch (what)
      {  case 'R':
            mn = glp_get_num_rows(lp); break;
         case 'C':
            mn = glp_get_num_cols(lp); break;
         default:
            insist(what != what);
      }
      for (ij = 1; ij <= mn; ij++)
      {  char *name;
         int len;
         if (what == 'R')
         {  name = glp_get_row_name(lp, ij);
            if (name == NULL) name = plain_name('R', ij);
         }
         else
         {  name = glp_get_col_name(lp, ij);
            if (name == NULL) name = plain_name('C', ij);
         }
         if (name[0] == '$') goto alas;
         len = strlen(name);
         if (len <= 8)
         {  strcpy(alias[ij], name);
            if (find_by_key(tree, alias[ij]) == NULL) goto fini;
            goto alas;
         }
         /* the first try: abc~wxyz */
         memcpy(alias[ij]+0, name+0, 3);
         memcpy(alias[ij]+3, "~", 1);
         memcpy(alias[ij]+4, name+(len-4), 4);
         if (find_by_key(tree, alias[ij]) == NULL) goto fini;
         /* the second try: abcd~xyz */
         memcpy(alias[ij]+0, name+0, 4);
         memcpy(alias[ij]+4, "~", 1);
         memcpy(alias[ij]+5, name+(len-3), 3);
         if (find_by_key(tree, alias[ij]) == NULL) goto fini;
         /* the third try: abcde~yz */
         memcpy(alias[ij]+0, name+0, 5);
         memcpy(alias[ij]+4, "~", 1);
         memcpy(alias[ij]+6, name+(len-2), 2);
         if (find_by_key(tree, alias[ij]) == NULL) goto fini;
         /* the fourth try: abcdef~z */
         memcpy(alias[ij]+0, name+0, 6);
         memcpy(alias[ij]+4, "~", 1);
         memcpy(alias[ij]+7, name+(len-1), 1);
         if (find_by_key(tree, alias[ij]) == NULL) goto fini;
alas:    /* hmm... nothing came of it :+( */
         strcpy(alias[ij], plain_name(what, ij));
         insist(find_by_key(tree, alias[ij]) == NULL);
fini:    /* enter the generated name to the symbol table */
         insist(strlen(alias[ij]) <= 8);
         insert_by_key(tree, alias[ij]);
      }
      delete_avl(tree);
      return;
}

/*----------------------------------------------------------------------
-- mps_number - convert number to standard 12-character MPS format.
--
-- This routine converts the given floating point value val to the
-- standard 12-character MPS format. It tries to provide maximal number
-- of significan digits. */

static char *mps_number(double val)
{     static char numb[255+1];
      int n;
      char *e;
      for (n = 12; n >= 6; n--)
      {  if (val != 0.0 && fabs(val) < 0.002)
            sprintf(numb, "%.*E", n, val);
         else
            sprintf(numb, "%.*G", n, val);
         insist(strlen(numb) <= 255);
         e = strrchr(numb, 'E');
         if (e != NULL) sprintf(e+1, "%d", atoi(e+1));
         if (strlen(numb) <= 12) return numb;
      }
      fault("glp_write_mps: can't convert floating point number '%g' to"
         " character string", val);
      return NULL; /* to relax compiler */
}

/* eof */
