/* glpchol/chol_symb.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpchol.h"

/*----------------------------------------------------------------------
-- chol_symb - compute Cholesky decomposition A = U'*U; symbolic phase
--
-- *Synopsis*
--
-- #include "glpchol.h"
-- void chol_symb(MAT *A, MAT *U, int head[], int next[], char work[]);
--
-- *Description*
--
-- The chol_symb routine implements symbolic phase of the Cholesky
-- decomposition A = U'*U, where A is a given sparse symmetric positive
-- definite matrix, U is resultant upper triangular matrix.
--
-- So far as the matrix A is symmetric, on entry it should contain only
-- upper triangular part. Only pattern of A is used, i.e. numeric values
-- of elements of the matrix A are ignored (they all are considered as
-- non-zeros). If A and U are different objects, the matrix A remains
-- unchanged on exit.
--
-- Order of the resultant matrix U should be the same as order of the
-- input matrix A. If A and U are different objects, on entry initial
-- contents of the matrix U is ignored. It is allowed that A and U are
-- the same object; in this case the matrix U is built in place of the
-- matrix A. (Note that the routine assigns the value +1.0 to elements
-- of the matrix U, which correspond to the original elements of the
-- matrix A, and the value -1.0 to elements, which appear as the result
-- of symbolic elimination. This may be used for visual estimating of
-- fill-in phenomenon.)
--
-- Auxiliary arrays head, next, and work should contain at least 1+n
-- elements, where n is order of the matrices A and U. If some of these
-- parameters is NULL, the chol_symb routine automatically allocates and
-- frees the corresponding working array(s).
--
-- *Method*
--
-- The chol_symb routine computes the matrix U in row-wise manner. No
-- pivoting is used, i.e. pivots are diagonal elements of the matrix A.
-- Before computing kth row the matrix U is the following:
--
--    1       k         n
-- 1  x x x x x x x x x x
--    . x x x x x x x x x
--    . . x x x x x x x x
--    . . . x x x x x x x
-- k  . . . . * * * * * *
--    . . . . . * * * * *
--    . . . . . . * * * *
--    . . . . . . . * * *
--    . . . . . . . . * *
-- n  . . . . . . . . . *
--
-- where 'x' denotes rows, which have been computed; '*' denotes rows
-- of the input matrix A (initially they are copied to the matrix U),
-- which have been not processed yet.
--
-- It is well known that in order to symbolically compute k-th row of
-- the matrix U it is necessary to merge k-th row of the matrix A (which
-- is placed in k-th row of the matrix U) and each i-th row of U, where
-- u[i,k] is non-zero (such rows are placed above k-th row).
--
-- However, in order to reduce number of rows, which should be merged,
-- the chol_symb routine uses the advanced algorithm proposed in:
--
-- D.J.Rose, R.E.Tarjan, and G.S.Lueker. Algorithmic aspects of vertex
-- elimination on graphs. SIAM J. Comput. 5, 1976, 266-83.
--
-- The authors of the cited paper show that we have the same result if
-- we merge k-th row of the matrix U and such its rows (among rows with
-- numbers 1, 2, ..., k-1), where leftmost non-zero non-diagonal element
-- is placed in k-th column. This feature signficantly reduces number of
-- rows, which should be merged, especially on the final steps, when the
-- computed rows of the matrix U becomes quite dense.
--
-- In order to determine row numbers, which should be merged with k-th
-- row of the matrix U, for a fixed time the routine uses linked lists
-- of row numbers. The location head[k] contains number of the first
-- row, where leftmost non-zero non-diagonal element is placed in k-th
-- column, and the location next[i] contains number of the next row in
-- the list, which has leftmost non-zero non-diagonal element in the
-- same column as i-th row. */

void chol_symb(MAT *A, MAT *U, int _head[], int _next[], char _work[])
{     ELEM *e;
      int n = A->m, i, j, k, *head = _head, *next = _next;
      char *work = _work;
      if (!(U->m == n && U->n == n && A->n == n))
         fault("chol_symb: inconsistent dimension");
      /* U := A */
      if (U != A) copy_mat(clear_mat(U), A);
      /* check that the original matrix is upper triangular and assign
         values +1.0 to the original non-zeros (this may be useful for
         visual estimating) */
      for (i = 1; i <= n; i++)
      {  for (e = U->row[i]; e != NULL; e = e->row)
         {  if (e->i > e->j)
               fault("chol_symb: input matrix is not upper triangular");
            e->val = +1.0;
         }
      }
      /* add diagonal elements (only those, which are missing) */
      for (i = 1; i <= n; i++)
      {  for (e = U->row[i]; e != NULL; e = e->row)
            if (e->i == e->j) break;
         if (e == NULL) new_elem(U, i, i, 0.0);
      }
      /* allocate and clear auxiliary arrays */
      if (_head == NULL) head = ucalloc(1+n, sizeof(int));
      if (_next == NULL) next = ucalloc(1+n, sizeof(int));
      if (_work == NULL) work = ucalloc(1+n, sizeof(char));
      for (i = 1; i <= n; i++) head[i] = next[i] = 0, work[i] = 0;
      /* main loop */
      for (k = 1; k <= n; k++)
      {  /* compute k-th row of the final matrix U */
         /* mark existing (original) elements of k-th row */
         for (e = U->row[k]; e != NULL; e = e->row) work[e->j] = 1;
         /* merge those rows of the matrix U, where leftmost non-zero
            non-diagonal element is placed in k-th column */
         for (i = head[k]; i != 0; i = next[i])
         {  insist(i < k);
            for (e = U->row[i]; e != NULL; e = e->row)
            {  /* ignore diagonal elements and elements, which already
                  exist in k-th row */
               if (e->i == e->j || work[e->j]) continue;
               /* create new non-zero in k-th row */
               new_elem(U, k, e->j, -1.0), work[e->j] = 1;
            }
         }
         /* k-th row has been symbolically computed */
         /* clear the array work and determine column index of leftmost
            non-zero non-diagonal element of k-th column */
         j = n+1;
         for (e = U->row[k]; e != NULL; e = e->row)
         {  insist(e->j >= k);
            if (e->i != e->j && j > e->j) j = e->j;
            work[e->j] = 0;
         }
         /* include k-th row into the appropriate linked list */
         if (j <= n) next[k] = head[j], head[j] = k;
      }
      /* free auxiliary arrays */
      if (_head == NULL) ufree(head);
      if (_next == NULL) ufree(next);
      if (_work == NULL) ufree(work);
      /* return to the calling program */
      return;
}

/* eof */
