/* glplp/create_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glplp.h"

/*----------------------------------------------------------------------
-- create_lp - create linear programming problem data block.
--
-- *Synopsis*
--
-- #include "glplp.h"
-- LP *create_lp(int m, int n, int mip);
--
-- *Description*
--
-- The create_lp routine creates a linear programming problem (LP) data
-- block, which describes the problem with m rows (auxiliary variables)
-- and n columns (structural variables).
--
-- The parameter mip is a flag. If it is zero, the routine creates data
-- block for pure LP problem. Otherwise, if this flag is non-zero, the
-- routine creates data block for MIP problem.
--
-- Initially all auxiliary variables are equal to zero, all structural
-- variables are non-negative, constraint matrix has no elements (i.e.
-- it is zero matrix), optimization direction is minimization, and all
-- coefficients of the objective function (including the constant term)
-- are equal to zero.
--
-- *Returns*
--
-- The create_lp routine returns a pointer to the created block. */

LP *create_lp(int m, int n, int mip)
{     LP *lp;
      int j, k;
      if (!(m > 0 && n > 0))
         fault("create_lp: invalid dimension");
      lp = umalloc(sizeof(LP));
      lp->m = m;
      lp->n = n;
      if (!mip)
         lp->kind = NULL;
      else
      {  lp->kind = ucalloc(1+n, sizeof(int));
         /* initially all structural variables are continuous */
         for (j = 1; j <= n; j++) lp->kind[j] = 0;
      }
      lp->type = ucalloc(1+m+n, sizeof(int));
      lp->lb = ucalloc(1+m+n, sizeof(double));
      lp->ub = ucalloc(1+m+n, sizeof(double));
      /* initially all auxiliary variables are equal to zero and all
         structural variables are non-negative */
      for (k = 1; k <= m+n; k++)
      {  lp->type[k] = (k <= m ? 'S' : 'L');
         lp->lb[k] = lp->ub[k] = 0.0;
      }
      lp->A = create_mat(m, n);
      lp->dir = '-';
      lp->c = ucalloc(1+n, sizeof(double));
      for (j = 0; j <= n; j++) lp->c[j] = 0.0;
      return lp;
}

/* eof */
