/* glprsm/crash_aa.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- crash_aa() - crash augmented constraint matrix.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int crash_aa(MAT *A, int tag[], PER *P, PER *Q);
--
-- *Description*
--
-- The routine crash_aa() tries to find an initial basis matrix using
-- the given augmented constraint matrix A and tags of "non-desirable"
-- columns.
--
-- On input the matrix A = (I | -A') is the augmented constraint matrix,
-- where I is the unity submatrix, columns of which correspond to the
-- auxiliary (logical) variables, and A' is the original constraint
-- matrix, columns of which correspond to the structural variables. Only
-- pattern of A is used, therefore A should contain no explicit zeros.
-- The matrix A is not changed on exit.
--
-- The array tag[] should have at least 1+n locations, where n is the
-- number of columns of the augmented matrix A (not A'). On input the
-- location tag[0] is not used, and the location tag[j], j = 1,...,n,
-- is a tag of the j-th column. If tag[j] is non-zero, this means that
-- it is *not* desirable to include the j-th column of the matrix A in
-- the initial basis matrix (such columns usually correspond to fixed
-- variables). The array tag[] is not changed on exit.
--
-- The result obtained by the routine is permutation matrices P and Q.
-- The routine finds such P and Q that the first m columns of the matrix
-- P*A*Q is a lower tringular matrix with non-zero diagonal, so this
-- submatrix may be used as an initial basis matrix. The routine tries
-- to minimize number of "non-desirable" columns in the initial basis
-- matrix. On input the matrices P and Q are ignored.
--
-- *Returns*
--
-- The routine crash_aa() returns the number of "non-desirable" columns,
-- which it couldn't get rid of and which are kept in the initial basis
-- matrix.
--
-- *Complexity*
--
-- The routine crash_aa() takes the time O(nz), where nz is number of
-- non-zeros in the matrix A.
--
-- *Algorithm*
--
-- The routine crash_aa() starts from the matrix W = P*A*Q, where P and
-- Q are the unity matrices, so initially W = A.
--
-- Before the next iteration W = (W1 | W2 | W3), where W1 is partially
-- built lower triangular submatrix, W2 is the active submatrix, and W3
-- is the submatrix that contains rejected columns. Graphically the
-- matrix W looks like the following:
--
--    1         k1         k2       n
-- 1  x . . . . . . . . . . . . . # #
--    x x . . . . . . . . . . . . # #
--    x x x . . . . . . . . . . # # #
--    x x x x . . . . . . . . . # # #
--    x x x x x . . . . . . . # # # #
-- k1 x x x x x * * * * * * * # # # #
--    x x x x x * * * * * * * # # # #
--    x x x x x * * * * * * * # # # #
--    x x x x x * * * * * * * # # # #
-- m  x x x x x * * * * * * * # # # #
--    <--W1---> <----W2-----> <--W3->
--
-- where the columns 1, ..., k1-1 form the submatrix W1, the columns
-- k1, ..., k2 form the submatrix W2, and the columns k2+1, ..., m form
-- the submatrix W3.
--
-- The matrix W has implicit representation. This means that actually
-- all transformations are performed on the permutation matrices P and
-- Q, which define the matrix W = P*A*Q.
--
-- Initially W1 and W3 are empty (have no columns), and W2 = A. Before
-- the first iteration the routine moves all "non-desirable" columns
-- from the active submatrix W2 to the submatrix W3 to prevent choosing
-- such columns on the subsequent iterations.
--
-- On each next iteration the routine looks for a singleton row, i.e.
-- the row that has the only non-zero element in the active submatrix
-- W2. If such a row exists and the corresponding element is w[i,j],
-- where (by the definition) k1 <= i <= m and k1 <= j <= k2, the routine
-- permutes k1-th and i-th rows and k1-th and j-th columns of the matrix
-- W in order to place the element in the position [k1,k1], removes the
-- k1-th column from the active submatrix W2, and includes this column
-- in the submatrix W1. If no singleton rows exist, but the active
-- submatrix W2 is not empty, the routine chooses a j-th column, whose
-- length is greatest in the active submatrix, removes this column from
-- W2, and includes it in W3, in the hope that rejecting the j-th column
-- will involve appearing new row singletons in W2.
--
-- If the active submatrix W2 becomes empty, the main phase is finished.
-- It may happen that the submatrix W1 has less than m columns, in which
-- case the routine padds the submatrix W1 to the lower triangular form
-- using the columns of the unity submatrix of the matrix A, which were
-- initially rejected as "non-desirable".
--
-- In order to find row signletons for a fixed time the Duff scheme rs
-- is used. This scheme implements a family {R(0), ..., R(n)} of rows of
-- the active submatrix, where R(len) is a set of rows that have len
-- non-zeros in the active submatrix. Each time when a column leaves the
-- active submatrix, the routine relocates each affected row from one
-- set to the other.
--
-- In order to find a column of the active submatrix, which has maximal
-- length, for a fixed time the Duff scheme cs is used. However, unlike
-- the scheme rs after initializing the scheme cs the routine scans this
-- scheme in the order of increasing column lengths and adds all columns
-- the the set C(0), which then is used as ordinary double linked list
-- to access columns in the reverse order.
--
-- Note that both schemes rs and cs hold rows and columns of the matrix
-- A, not the matrix W. */

struct csa
{     /* common storage area */
      MAT *A; /* MAT A[1:m,1:n]; */
      /* augmented constraint matrix A = (I | -A'), where A' is the
         original constraint matrix */
      PER *P; /* PER P[1:m]; */
      /* the row permutation matrix */
      PER *Q; /* PER Q[1:n]; */
      /* the column permutation matrix */
      int k1;
      /* the leftmost column of the active submatrix W2 */
      int k2;
      /* the rightmost column of the active submatrix W2 */
      DUFF *rs;
      /* active rows of the matrix A */
      DUFF *cs;
      /* active columns of the matrix A */
};

#define iW(i) (P->col[i])
/* converts row number of A to row number of W */

#define iA(i) (P->row[i])
/* converts row number of W to row number of A */

#define jW(j) (Q->row[j])
/* converts column number of A to column number of W */

#define jA(j) (Q->col[j])
/* converts column number of W to column number of A */

/*----------------------------------------------------------------------
-- remove_col() - remove column from the active submatrix.
--
-- This routine updates the row and column Duff schemes for subsequent
-- removing the j-th column from the active submatrix W2. The number j
-- corresponds to the matrix W, not to the matrix A. */

static void remove_col(struct csa *csa, int j)
{     MAT *A = csa->A;
      PER *Q = csa->Q;
      DUFF *rs = csa->rs, *cs = csa->cs;
      int k1 = csa->k1, k2 = csa->k2, len;
      ELEM *e;
      insist(k1 <= j && j <= k2);
      /* update the row scheme */
      for (e = A->col[jA(j)]; e != NULL; e = e->col)
      {  len = rs->len[e->i];
         exclude_obj(rs, e->i);
         include_obj(rs, e->i, len-1);
      }
      /* remove the j-th column from the active column list */
      exclude_obj(cs, jA(j));
      return;
}

/*----------------------------------------------------------------------
-- permute_rows() - permute rows of the matrix W.
--
-- This routine permutes i1-th and i2-th rows of the matrix W. */

static void permute_rows(struct csa *csa, int i1, int i2)
{     PER *P = csa->P;
      int t1, t2;
      t1 = iA(i1), t2 = iA(i2);
      iA(i1) = t2, iW(t2) = i1;
      iA(i2) = t1, iW(t1) = i2;
      return;
}

/*----------------------------------------------------------------------
-- permute_cols() - permute columns of the matrix W.
--
-- This routine permutes j1-th and j2-th columns of the matrix W. */

static void permute_cols(struct csa *csa, int j1, int j2)
{     PER *Q = csa->Q;
      int t1, t2;
      t1 = jA(j1), t2 = jA(j2);
      jA(j1) = t2, jW(t2) = j1;
      jA(j2) = t1, jW(t1) = j2;
      return;
}

/*----------------------------------------------------------------------
-- crash_aa() - crash augmented constraint matrix.
--
-- This routine finds an initial basis matrix using the given augmented
-- constraint matrix A and the tags of "non-desirable" columns. */

int crash_aa(MAT *A, int tag[], PER *P, PER *Q)
{     struct csa _csa, *csa = &_csa;
      DUFF *rs, *cs;
      ELEM *e;
      int m, n, i, j, len, ret;
      m = A->m, n = A->n;
      insist(P->n == m && Q->n == n);
      /* reset permutation matrices, P := I, Q := I */
      reset_per(P);
      reset_per(Q);
      /* create and initialize Duff schemes */
      rs = create_duff(m, n);
      cs = create_duff(n, m);
      for (i = 1; i <= m; i++)
         include_obj(rs, i, count_nz(A, +i));
      for (j = 1; j <= n; j++)
         include_obj(cs, j, count_nz(A, -j));
      /* add all columns of the matrix A to the set C(0) in the order
         of increasing their lengths; these columns will be scanned in
         the reverse (decreasing) order */
      for (len = 1; len <= m; len++)
      {  while (cs->head[len] != 0)
         {  j = cs->head[len];
            exclude_obj(cs, j);
            include_obj(cs, j, 0);
         }
      }
      /* initialize csa; set initial edges of the active submatrix */
      csa->A = A, csa->P = P, csa->Q = Q;
      csa->k1 = 1, csa->k2 = n;
      csa->rs = rs, csa->cs = cs;
      /* now W1 = W3 = <empty>, W2 = W = A */
      /* remove "non-desirable" columns from W2 to W3 */
      for (j = csa->k1; j <= csa->k2; )
      {  if (tag[jA(j)])
         {  remove_col(csa, j);
            permute_cols(csa, csa->k2, j);
            csa->k2--;
         }
         else
            j++;
      }
      /* the main loop starts here */
      while (csa->k1 <= csa->k2)
      {  if (rs->head[1] != 0)
         {  /* the active submatrix W2 has a row singleton */
            i = iW(rs->head[1]);
            /* the singleton in the i-th row of W; find it */
            j = 0;
            for (e = A->row[iA(i)]; e != NULL; e = e->row)
            {  if (csa->k1 <= jW(e->j) && jW(e->j) <= csa->k2)
               {  insist(j == 0);
                  j = jW(e->j);
               }
            }
            insist(j != 0);
            /* the singleton has been found in j-th column of W */
            /* remove the j-th column from the active submatrix W2 */
            remove_col(csa, j);
            /* move w[i,j] to the position w[k1,k1] and add the k1-th
               column of W to the submatrix W1 */
            permute_rows(csa, csa->k1, i);
            permute_cols(csa, csa->k1, j);
            csa->k1++;
         }
         else
         {  /* the active submatrix has no row singletons */
            /* if there is no more columns in the active submatrix W2,
               terminate the process */
            if (cs->head[0] == 0) break;
            /* pull the first column from C(0); this column has largest
               length among other active columns */
            j = jW(cs->head[0]);
            insist(csa->k1 <= j && j <= csa->k2);
            /* remove the j-th column from the active submatrix W2 and
               drop it out adding to the submatrix W3 */
            remove_col(csa, j);
            permute_cols(csa, csa->k2, j);
            csa->k2--;
         }
      }
      /* now there is no "non-desirable" columns in the submatrix W1,
         so it's time to determine its rank deficit */
      ret = m - csa->k1 + 1;
      /* padd the submatrix W1 to the lower triangular form by columns
         of the unity submatrix of the matrix A */
      while (csa->k1 <= m)
      {  /* k1-th column of the matrix W should have 1 in the position
            (k1,k1); but the k1-th row of the matrix W corresponds to
            the iA(k1)-th row of the matrix A, that allows to determine
            what column of the matrix A should be placed instead the
            k1-th column of the matrix W */
         j = jW(iA(csa->k1));
         insist(j >= csa->k2);
         permute_cols(csa, csa->k1, j);
         csa->k1++;
      }
      /* free auxiliary data structure */
      delete_duff(rs);
      delete_duff(cs);
      /* returns to the calling program */
      return ret;
}

/* eof */
