/* glprsm/ftran.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- ftran - perform forward transformation.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void ftran(RSM *rsm, double u[], int save);
--
-- *Description*
--
-- The ftran routine performs forward transformation (FTRAN) of the
-- vector u using some representation of the current basis matrix.
--
-- This operation means solving the system B*u' = u, where B is the
-- current basis matrix, u is the given vector which should be
-- transformed, u' is the resultant vector.
--
-- On entry the array u should contain elements of the vector u in
-- locations u[1], u[2], ..., u[m], where m is order of the matrix B.
-- On exit this array contains elements of the vector u' in the same
-- locations.
--
-- The parameter save is a flag. If this flag is set, it means that the
-- vector u is the column of non-basic variable (xN)q which has been
-- chosen to enter the basis (i.e. u = Nq).In this case the ftran
-- routine saves some internal information which will be used further
-- by the update_b routine in order to update the representation of the
-- basis matrix for the adjacent basis. It is assumed that the calling
-- program should perform at least one call to the ftran routine with
-- the save flag set before subsequent call to the update_b routine. */

void ftran(RSM *rsm, double u[], int save)
{     if (rsm->pfi != NULL)
         pfi_ftran(rsm->pfi, u, save);
      else if (rsm->rfi != NULL)
         rfi_ftran(rsm->rfi, u, save);
      else if (rsm->afi != NULL)
         afi_ftran(rsm->afi, u, save);
      else if (rsm->ufi != NULL)
         ufi_ftran(rsm->ufi, u, save);
      else
         insist(rsm != rsm);
      return;
}

/* eof */
