/* glprsm/rsm_dual.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_dual - find optimal solution using dual simplex method.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_dual(RSM *rsm, int (*monit)(void), double c[],
--    double tol_bnd, double tol_dj, double tol_piv, double dvec[],
--    int relax);
--
-- *Description*
--
-- The rsm_dual routine searches for optimal solution of LP problem
-- using the dual simplex method.
--
-- The parameter rsm points to the common block. On entry this block
-- specifies an initial basis solution which should be dual feasible.
-- On exit this block specifies the last reached basis solution.
--
-- The parameter monit specifies the user-supplied routine used for
-- monitoring purposes. It is called by the rsm_dual routine each time
-- before the next iteration. If the monit routine returns non-zero,
-- the search is terminated. The parameter monit can be set to NULL.
--
-- The array c specifies elements of the expanded vector of coefficients
-- of the objective function in locations c[1], ..., c[m+n]. This array
-- is not changed on exit. Should note that the routine *minimizes* the
-- objective function, therefore in the case of maximization the vector
-- c should have the opposite sign.
--
-- The parameter tol_bnd is a relative tolerance which is used by the
-- routine in order to see if the current solution is primal feasible.
--
-- The parameter tol_dj is a relative tolerance which is used by the
-- routine in order to see if the current solution is dual feasible.
--
-- The parameter tol_piv is a relative tolerance which is used by the
-- routine in order to choose the pivot element of the simplex table.
--
-- If the parameter dvec is NULL, the routine chooses basic variable
-- using the standard technique. Otherwise the routine uses the steepest
-- edge technique. In the latter case the array dvec should specifiy
-- elements of the vector delta in locations dvec[1], ..., dvec[m]. On
-- entry the vector delta should correspond to the initial basis passed
-- to the routine. This vector is updated on every iteration, therefore
-- on exit it corresponds to the final basis reached by the routine.
--
-- The parameter relax is a flag. If this flag is zero, the routine
-- chooses non-basic variable using the standard ratio test. Otherwise
-- the routine uses two-pass ratio test.
--
-- *Returns*
--
-- The rsm_dual routine returns one of the following codes:
--
-- 0 - optimal solution found;
-- 1 - problem has no (primal) feasible solution;
-- 2 - numerical stability lost (the current basis solution became dual
--     infeasible due to round-off errors);
-- 3 - numerical problems with basis matrix (the current basis matrix
--     became singular or ill-conditioned due to unsuccessful choice of
--     the pivot element on the last simplex iteration);
-- 4 - user-supplied routine returned non-zero code. */

static int debug = 0; /* debug mode flag (to check the vector delta) */

int rsm_dual(RSM *rsm, int (*monit)(void), double c[],
      double tol_bnd, double tol_dj, double tol_piv, double dvec[],
      int relax)
{     int m = rsm->m, n = rsm->n, p, tagp, q, ret;
      double *bbar, *pi, *cbar, *ap, *aq, *zeta;
      /* check common block for correctness */
      check_rsm(rsm);
      /* allocate working arrays */
      bbar = ucalloc(1+m, sizeof(double));
      pi = ucalloc(1+m, sizeof(double));
      cbar = ucalloc(1+n, sizeof(double));
      ap = ucalloc(1+n, sizeof(double));
      aq = ucalloc(1+m, sizeof(double));
      zeta = ucalloc(1+m, sizeof(double));
      /* main loop starts here */
      for (;;)
      {  /* call user-supplied routine */
         if (monit != NULL)
         {  if (monit())
            {  ret = 4;
               break;
            }
         }
         /* compute current values of basic variables */
         eval_bbar(rsm, bbar);
         /* compute simplex multipliers */
         eval_pi(rsm, c, pi);
         /* compute reduced costs of non-basic variables */
         eval_cbar(rsm, c, pi, cbar);
         /* current basis solution should be dual feasible */
         if (check_cbar(rsm, c, cbar, tol_dj))
         {  ret = 2;
            break;
         }
         /* choose basic variable xB[p] */
         p = dual_row(rsm, bbar, dvec, &tagp, tol_bnd);
         if (p == 0)
         {  /* optimal solution found */
            ret = 0;
            break;
         }
         /* compute p-th row of inv(B) */
         eval_zeta(rsm, p, zeta);
         /* compute p-th (pivot) row of the simplex table */
         eval_row(rsm, zeta, ap);
         /* choose non-basic variable xN[q] */
         if (!relax)
         {  /* use standard ratio test */
            q = dual_col(rsm, tagp, ap, cbar, tol_piv);
         }
         else
         {  /* use two-pass ratio test */
            q = harris_col(rsm, tagp, ap, c, cbar, tol_piv,
#if 0
               0.15 * tol_dj);
#else /* 3.0.3 */
               0.003 * tol_dj);
#endif
         }
         if (q == 0)
         {  /* problem has no (primal) feasible solution */
            ret = 1;
            break;
         }
         /* compute q-th (pivot) column of the simplex table */
         eval_col(rsm, q, aq, 1);
         /* update the vector delta */
         if (dvec != NULL)
            update_dvec(rsm, dvec, p, q, ap, aq, zeta);
         /* correct tagp if xB[p] is fixed variable */
         if (rsm->type[rsm->indb[p]] == 'S') tagp = 'S';
         /* jump to the adjacent basis */
         if (change_b(rsm, p, tagp, q))
         {  /* numerical problems with basis matrix */
            ret = 3;
            break;
         }
         /* check accuracy of the vector delta */
         if (debug && dvec != NULL)
            print("check_dvec: %g", check_dvec(rsm, dvec));
         /* end of main loop */
      }
      /* free working arrays */
      ufree(bbar);
      ufree(pi);
      ufree(cbar);
      ufree(ap);
      ufree(aq);
      ufree(zeta);
      /* return to the calling program */
      return ret;
}

/* eof */
