/* glpset/get_atom.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpset.h"

/*----------------------------------------------------------------------
-- get_atom - allocate atom of fixed size.
--
-- *Synopsis*
--
-- #include "glpset.h"
-- void *get_atom(POOL *pool);
--
-- *Description*
--
-- The get_atom routine allocates an atom (i.e. continuous space of
-- memory) using the specified memory pool. The size of atom which
-- should be allocated is assumed to be set when the pool was created
-- by the create_pool routine.
--
-- Note that being allocated the atom initially contains arbitrary data
-- (not binary zeros).
--
-- *Returns*
--
-- The get_atom routine returns a pointer to the allocated atom. */

void *get_atom(POOL *pool)
{     void *ptr;
      if (pool->size == 0)
         fault("get_atom: pool cannot be used to allocate an atom of fi"
            "xed size");
      /* if there is a free atom, pull it from the list */
      if (pool->avail != NULL)
      {  ptr = pool->avail;
         pool->avail = *(void **)ptr;
         goto done;
      }
      /* free atom list is empty; if the last allocated block does not
         exist or if it has not enough space, we need a new block */
      if (pool->link == NULL || pool->used + pool->size > POOL_SIZE)
      {  /* we can pull a new block from the list of free blocks, or if
            this list is empty, we need to allocate such block */
         if (pool->stock != NULL)
         {  ptr = pool->stock;
            pool->stock = *(void **)ptr;
         }
         else
            ptr = umalloc(POOL_SIZE);
         /* the new block becomes the last allocated block */
         *(void **)ptr = pool->link;
         pool->link = ptr;
         /* now only few bytes in the new block are used to hold a
            pointer to the previous allocated block */
         pool->used = align_datasize(sizeof(void *));
      }
      /* the last allocated block exists and it has enough space to
         allocate an atom */
      ptr = (void *)((char *)pool->link + pool->used);
      pool->used += pool->size;
done: pool->count++;
#if 0
      memset(ptr, '\0', pool->size);
#else
      memset(ptr, '?', pool->size);
#endif
      return ptr;
}

/* eof */
