/* cmd-whats-missing.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "tla/libfsutils/rmrf.h"
#include "tla/libarch/namespace.h"
#include "tla/libarch/project-tree.h"
#include "tla/libarch/patch-logs.h"
#include "tla/libarch/local-cache.h"
#include "tla/libarch/make-changeset.h"
#include "tla/libarch/archive.h"
#include "tla/libarch/whats-missing.h"
#include "tla/libarch/changeset-report.h"
#include "tla/libarch/cmd-whats-missing.h"



static t_uchar * usage = "[options] [revision]";
static t_uchar * version_string = (cfg__std__package " from regexps.com\n"
                                   "\n"
                                   "Copyright 2003 Tom Lord\n"
                                   "\n"
                                   "This is free software; see the source for copying conditions.\n"
                                   "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
                                   "PARTICULAR PURPOSE.\n"
                                   "\n"
                                   "Report bugs to " cfg__tla_bug_mail ".\n"
                                   "\n");

#define OPTS(OP, OP2) \
  OP (opt_help_msg, "h", "help", 0, \
      "Display a help message and exit.") \
  OP (opt_long_help, "H", 0, 0, \
      "Display a verbose help message and exit.") \
  OP (opt_version, "V", "version", 0, \
      "Display a release identifier string") \
  OP2 (opt_version, 0, 0, 0, "and exit.") \
  OP (opt_archive, "A", "archive", 1, \
      "Override `my-default-archive'") \
  OP (opt_dir, "d", "dir DIR", 1, \
      "cd to DIR first") \
  OP (opt_reverse, "r", "reverse", 0, \
      "sort from newest to oldest") \
  OP (opt_summary, "s", "summary", 0, \
      "display a summary of each missing patch") \
  OP (opt_creator, "c", "creator", 0, \
      "display the creator of each missing patch") \
  OP (opt_date, "D", "date", 0, \
      "display the date of each missing patch") \
  OP (opt_full, "f", "full", 0, \
      "print full revision names") \
  OP (opt_merges, 0, "merges", 0, \
      "print a merge list for each missing patch")

t_uchar arch_cmd_whats_missing_help[] = ("print patches missing from a project tree\n"
                                         "Print a list of patches missing in the project tree containing\n"
                                         "DIR (or the current directory) for VERSION (or the default version.\n"
                                         "of the project tree).\n"
                                         "\n"
                                         "The flag --merges means, for each patch, to print the list of patches\n"
                                         "included in the patch in two columns.  For example:\n"
                                         "\n"
                                         "        PATCH-A        PATCH-A\n"
                                         "        PATCH-A        PATCH-B\n"
                                         "        PATCH-A        PATCH-C\n"
                                         "\n"
                                         "means that PATCH-A includes the changes from PATCH-B and PATCH-C.\n"
                                         "(Every patch includes at least itself.)\n");

enum options
{
  OPTS (OPT_ENUM, OPT_IGN)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC, OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_whats_missing (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  t_uchar * dir = 0;
  int reverse = 0;
  int summarized_headers = 0;
  int full = 0;
  int merges = 0;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, version_string, arch_cmd_whats_missing_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;


        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_dir:
          {
            lim_free (0, dir);
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_reverse:
          {
            reverse = 1;
            break;
          }
        case opt_summary:
          {
            summarized_headers |= arch_include_summary;
            break;
          }
        case opt_creator:
          {
            summarized_headers |= arch_include_creator;
            break;
          }
        case opt_date:
          {
            summarized_headers |= arch_include_date;
            break;
          }
        case opt_merges:
          {
            merges = 1;
            break;
          }
        }
    }

  if (argc > 2)
    goto usage_error;

  {
    t_uchar * tree_root = 0;
    t_uchar * vsnspec = 0;
    t_uchar * archive = 0;
    t_uchar * version= 0;
    struct arch_archive * arch = 0;
    rel_table whats_missing = 0;

    if (default_archive && !arch_valid_archive_name (default_archive))
      {
        safe_printfmt (2, "whats-missing: invalid archive name: %s\n", default_archive);
        exit (1);
      }

    tree_root = arch_tree_root (0, dir, 0);

    if (!tree_root)
      {
        safe_printfmt (2, "whats-missing: not in a project tree\n  dir: %s\n", dir);
        exit (1);
      }

    if (argc == 2)
      vsnspec = str_save (0, argv[1]);
    else
      {
        vsnspec = arch_tree_version (tree_root);
        if (!vsnspec)
          {
            safe_printfmt (2, "whats-missing: tree has no default version\n  tree: %s\n", tree_root);
            exit (1);
          }
      }

    if (!arch_valid_package_name (vsnspec, arch_maybe_archive, arch_req_version, 0))
      {
        safe_printfmt (2, "whats-missing: illegal version name: %s\n", vsnspec);
        exit (2);
      }

    archive = arch_parse_package_name (arch_ret_archive, default_archive, vsnspec);
    version = arch_parse_package_name (arch_ret_non_archive, 0, vsnspec);


    arch = arch_archive_connect (archive, 0);

    whats_missing = arch_whats_missing (tree_root, arch, version);

    if (reverse)
      arch_sort_table_by_patch_level_field (1, whats_missing, 0);

    {
        int x;

        for (x = 0; x < rel_n_records (whats_missing); ++x)
          {
            if (!summarized_headers)
              {
                if (full)
                  safe_printfmt (1, "%s/%s--%s\n", archive, version, whats_missing[x][0]);
                else
                  safe_printfmt (1, "%s\n", whats_missing[x][0]);
              }
            else
              {
                t_uchar * revision = 0;
                t_uchar * log = 0;
                t_uchar * body = 0;
                assoc_table headers = 0;

                revision = str_alloc_cat_many (0, version, "--", whats_missing[x][0], str_end);
                log = arch_archive_log (arch, revision);

                arch_parse_log (0, &headers, &body, log);

                if (!merges)
                  {
                    if (full)
                      safe_printfmt (1, "%s/%s--%s\n", archive, version, whats_missing[x][0]);
                    else
                      safe_printfmt (1, "%s\n", whats_missing[x][0]);
                  }
                else
                  {
                    panic ("--merge not yet");
                  }


                arch_print_headers_summary (1, 4, headers, summarized_headers);

                lim_free (0, revision);
                lim_free (0, log);
                lim_free (0, body);
                free_assoc_table (headers);
              }
          }
      }

    arch_archive_close (arch);

    lim_free (0, vsnspec);
    lim_free (0, tree_root);
    lim_free (0, archive);
    lim_free (0, version);
    rel_free_table (whats_missing);
  }


  lim_free (0, dir);
  lim_free (0, default_archive);

  return 0;
}



/* tag: Tom Lord Sat May 24 23:36:40 2003 (whats-missing.c)
 */
