/*
Copyright (C) 1995 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the 
terms of the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef _mMatrix_h_
#define _mMatrix_h_

#include<mVector.h>

#define _START_ 1
#define _COMP_ <=

template<class C> class Delay_rc;

template<class T,class S = allocator<T> >
class mMatrix
{
// zero offset for now, stored in row major form only, assume S is an
// iterator type, submatrices will have to be generated as a mask at the 
// container level.
public:
  typedef T                     value_type;
  typedef value_type&           reference;
  typedef const value_type&     const_reference;
  typedef S::size_type		size_type;
  typedef S::pointer            container_type;
  typedef mVector<T,S>		array_type;
  typedef Delay_rc<S>		d_type;
  typedef mVector<T, rc_allocator<S> > 	rc_type;

  mMatrix() : row_(0), col_(0), diag_(0) {}

  mMatrix(size_type ri,size_type ci,const_reference a=value_type()) :\
r_(ri), c_(ci), a_(ri*ci,a), row_(0), col_(0), diag_(0) {}

  mMatrix(const mMatrix &v) : r_(v.r_), c_(v.c_), a_(v.a_), row_(0), col_(0), diag_(0) {}

#ifdef _MEMBER_TEMPLATES_
  template<class T2,class S2>
  mMatrix(const mVector<T2,S2> &v)
#else          NOT _MEMBER_TEMPLATES_
  mMatrix(const mVector<T,S> &v)
#endif          NOT _MEMBER_TEMPLATES_
: r_(v.size()), c_(v.size()), a_(v.size()*v.size()), row_(0), col_(0), diag_(0) {
    for(size_type j=1;j<=r_;j++) (*this)(j,j)=v(j); }

  ~mMatrix() { rc_dealloc(); }

  reference operator()(size_type ri,size_type ci) {
    return (reference)a_(c_*(ri-1)+ci); }
  value_type operator()(size_type ri,size_type ci) const {
    return a_(c_*(ri-1)+ci); }

  container_type& container() { return a_.container(); }
  const container_type& container() const { return a_.container(); }

  array_type& array() { return a_; }
  const array_type& array() const { return a_; }

  rc_type& row(size_type r) { 
    if(!row_) {
      row_=(new (rc_type*)[r_])-1;
      for(size_t j=1;j<=r_;j++) row_[j]=(rc_type*)0;
    }
    if(!row_[r]) {
      row_[r] = new rc_type;
      (row_[r])->resize(c_);
      (row_[r])->container()=d_type(a_.container(),(r-1)*c_,1);
    }
    return *(row_[r]); }
  const rc_type row(size_type r) const { 
    rc_type row;
    row.resize(c_);
    row.container()=d_type(a_.container(),(r-1)*c_,1); 
    return row; }

  rc_type& col(size_type c) { 
    if(!col_) {
      col_=(new (rc_type*)[c_])-1;
      for(size_t j=1;j<=c_;j++) col_[j]=(rc_type*)0;
    }
    if(!col_[c]) {
      col_[c] = new rc_type;
      (col_[c])->resize(r_);
      (col_[c])->container()=d_type(a_.container(),(c-c_),c_);
    }
    return *(col_[c]); }
  const rc_type col(size_type c) const { 
    rc_type col;
    col.resize(r_);
    col.container()=d_type(a_.container(),(c-c_),c_);
    return col; }

  rc_type& diagonal() { 
    if(!diag_) {
      diag_ = new rc_type;
      diag_->resize(min(r_,c_));
      diag_->container()=d_type(a_.container(),-c_,c_+1);
    }
    return *(diag_); }
  const rc_type diagonal() const { 
    rc_type diag;
    diag.resize(min(r_,c_));
    diag.container()=d_type(a_.container(),-c_,c_+1);
    return diag; }

  size_type rsize() const { return r_; }
  size_type csize() const { return c_; }
  void resize(size_type ri,size_type ci) { 
    rc_dealloc(); r_=ri; c_=ci; a_.resize(ri*ci); }
  void reserve(size_type ri,size_type ci,const_reference b=value_type()) { 
    a_.reserve(ri*ci,b); resize(ri,ci); }

  mMatrix& operator=(const_reference b) {
    for(size_type ri=1; ri<=rsize(); ri++)
      for(size_type ci=1; ci<=csize(); ci++) (*this)(ri,ci)=b;
    return (*this); }
#ifdef _MEMBER_TEMPLATES_
  template<class T2,class S2>
  mMatrix& operator=(const mMatrix<T2,S2>& b) {
#else          NOT _MEMBER_TEMPLATES_
  mMatrix& operator=(const mMatrix& b) {
#endif          NOT _MEMBER_TEMPLATES_
    for(size_type ri=1; ri<=rsize(); ri++)
      for(size_type ci=1; ci<=csize(); ci++) (*this)(ri,ci)=b(ri,ci);
    return (*this); }

  mMatrix& transpose();

private:
  rc_type** row_;
  rc_type** col_;
  rc_type* diag_;
  array_type a_;
  size_type r_,c_;
  void rc_dealloc();
};        // End of class mMatrix

#include"matrix_def.h"

#undef _START_
#undef _COMP_

#endif       _mMatrix_h_

