// This file defines several classes which implement expansion functions
// for discrete transforms and collocation problems.

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This file is part of the GNUSSL software package.  This package is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this distribution; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

template<class T>
class Colloc
{
// Virtual class which outlines the interface for a set of collocation
// or expansion functions
public:
  typedef mMatrix<T,allocator<T> >		dMat;   // derivative matrix
  typedef dMat*					dMpntr; // pointer to dMat
  typedef mArray<dMpntr,allocator<dMpntr> >	dArr;   // array of dMat's

  Colloc() : a_((T)0.0), b_((T)0.0), order_(0), length_(0) {}

  T& a() { return a_; }                       // endpoints of mesh
  T& b() { return b_; }                       // endpoints of mesh
  size_t order() const { return order_; }     // number of expansion functions
  size_t length() const { return length_; }   // number of mesh points

  virtual resize(size_t l,size_t o) {         // reset length and order
    for(size_t j=0;j<dA_.size();j++) delete dA_(j);   // clean up matrices
    order_=o;
    length_=l; }

  virtual T operator()(size_t j,const T& x) = 0;  // j'th function at x
  virtual T mesh(size_t j) {       // j'th default mesh point, even distribution
    return a()+(b()-a())*(j-1.0)/(length()-1.0); }

  virtual T derivMatElem(size_t r,size_t c) { return (T)0.0; }
  // returns the elements of the matrix which, when dotted on the left into
  // a vector of function coefficients, yields the vector of coefficients
  // of the first derivative of the original function

  dMat& derivMat(size_t deriv) {
  // returns a reference to a stored matrix which is the d'th power of the 
  // first derivative matrix
    if(deriv>dA_.size()-1) {                  // need to increase array size
      size_t oldsize=dA_.size();
      dArr tdA(deriv+1);                       // alloc temporary
      for(int j=0;j<oldsize;j++) tdA=dA_(j);   // copy pointers
      dA_.resize(deriv+1);                     // reallocate storage array
      for(int j=0;j<oldsize;j++) dA_(j)=tdA(j);// copy pointers back
      for(int j=oldsize+1;j<=deriv;j++) {      // and allocate new derivatives
        dA_(j)=new dMat;
        dA_(j)->resize(order(),order());
      }
      if(oldsize<1) dA_(0)->diagonal()=(T)1.0; // init the identity
      if(oldsize<2 && deriv>=1)                // init the first deriv
        for(int r=1;r<=order();r++)
          for(int c=1;c<=order();c++) dA_(1)->operator()(r,c)=derivMatElem(r,c);
            // and compute the remaining uninitialized matrices
      for(int j=oldsize+1;j<=deriv;j++) dot(*(dA_(j-1)),*(dA_(1)),*(dA_(j)));
    }
    return *(dA_(deriv)); }

protected:
  T a_,b_;                                     // endpoints of mesh
  size_t order_;                               // number of expansion functions
  size_t length_;                              // number of mesh points
private:
  dArr dA_;                         // array of pointers to derivative matrices
};

template<class T>
class Monomial : public Colloc<T>
{     // implements monomial powers of x for transform and fitting classes
public:
  T operator()(size_t index,const T& x) { return pow(x,(T)(index-1)); }
  T derivMatElem(size_t r,size_t c) {
    if(c!=r+1) return (T)0.0;
    return (T)r; }
}; 


template<class T>
class ConstrainedPolynomial : public Colloc<T>
{
// implements polynomials of x, constrained by specifying the function 
// values and derivatives at some point

ConstrainedPolynomial() { 
  cerr << "ConstrainedPolynomial: NOT IMPLEMENTED YET\n";
  exit(1);
}

};

template<class T>
class Chebyshev : public Colloc<T>
{
// Implements Chebyshev polynomials
public:
  Chebyshev() { a()=(T)-1.0; b()=(T)1.0; }

  T operator()(size_t n,const T& x) {        // n'th fnc at point x
    // Returns the value of the (n-1)'th Chebyshev polynomial at x in [a,b].
    n-=1;     // translate from Transform indexing to "natural" indexing
    if(n==0) return (T)0.5;
    x=-1+2.0*(x-a())/(b()-a());           // rescale x to [-1,1]
    if(n==1) return x;
    T t,tnm1=1,tn=x;
    for(size_t j=2;j<=n;j++) {
      t=2.0*x*tn - tnm1;          // use the direct recurrence relation
      tnm1=tn;
      tn=t;
    }
    return t; }

  T mesh(size_t j) {
    // Returns the position of the j'th Chebyshev collocation point out of
    // length points (including first and last), spanning the interval [a,b].
    // We have chosen the extrema of the (length+1)'th Chebyshev polynomial.
    // The zeroes have the advantage of a discrete orthogonality condition, but
    // maxima have the advantage that they include the endpoints of the interval
    // so that boundary conditions may be included.  (pnt(1)=a, pnt(length)=b)
    return (a() + 0.5*(1.0 - cos(M_PI*(j-1.0)/(length-1.0)))*(b()-a())); }

  T derivMatElem(int r,int c) {  // deriv matrix element (r,c)
    // returns the (r,c) value of the derivative matrix.  
    if(c<=r) return 0;
    if((c+r)%2==0) return 0;
    double t=4*(c-1)/(b-a);
    return t; }
};

template<class T>
class ChebyMax : public Chebyshev<T>
{
public:
  ChebyMax() {}

  T mesh(size_t j) {
    // Returns the position of the j'th Chebyshev collocation point out of
    // length points (including first and last), spanning the interval [a,b].
    // This implements the extrema of the (length()+1)'th Chebyshev polynomial.
    return (a() + 0.5*(1.0 - cos(M_PI*(j-1.0)/(length()-1)))*(b()-a())); }
};

template<class T>
class ChebyZero : public Chebyshev<T>
{
public:
  ChebyZero() {}

  T mesh(size_t j) {
    // Returns the position of the j'th Chebyshev collocation point out of
    // length total points, spanning the interval (a,b).
    // This implements the zeroes of the (length()+1)'th Chebyshev polynomial.
    return (a() + 0.5*(1.0 - cos(M_PI*(j-0.5)/length()))*(b()-a())); }
};

template<class Complex>
class Fourier : public Colloc<Complex>
{
// Implements complex exponentials for discrete Fourier expansions
public:
  Fourier() { a()=Complex(0.0,0.0); b()=Complex(2.0*M_PI,0.0); }
  Complex operator()(size_t n,const Complex& x) {
    Complex t(0.0,-2.0*M_PI*(n-1));
    return exp(t*(x-a())/(b()-a()))/sqrt(length()); }

  Complex mesh(size_t j) { return (a() + (j-1)*(b()-a())/length()); }

  Complex derivMatElem(size_t r,size_t c) { 
    if(r==c) return Complex(0.0,-2.0*M_PI*(c-1))/(b()-a());
    return Complex(0.0,0.0); }
};

template<class Real>
class FourierSin : public Colloc<Real>
{
// Implements sine for discrete Fourier expansions
public:
  FourierSin() { a()=(Real)0.0; b()=(Real)M_PI; }
  Real operator()(size_t n,const Real& x) {
    return sqrt(2.0/(length()+1))*sin(M_PI*n*(x-a())/(b()-a())); }

  Real mesh(size_t j) { return (a() + j*(b()-a())/(length()+1)); }
};

template<class Real>
class FourierCos : public Colloc<Real>
{
// Implements cosine for discrete Fourier expansions
public:
  FourierCos() { a()=(Real)0.0; b()=(Real)M_PI; }
  Real operator()(size_t n,const Real& x) {
    if(n==1) return 0.5*sqrt(2.0/(length()-1));
    Real tmp= sqrt(2.0/(length()-1))*cos(M_PI*(n-1)*(x-a())/(b()-a())); 
    if(n==order()) return 0.5*tmp;
    return tmp; }

  Real mesh(size_t j) { return (a() + (j-1)*(b()-a())/(length()-1)); }

};

