// This file defines all the functions which are included in the C-library,
// libgnumath.a .

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This file is part of the GNUSSL software package.  This package is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this distribution; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include"c_complex.h"
extern "C" {
#include"c_array.h"
}
#include<lin_alg.h>

typedef struct c_double_vector cdv_;            // short hand double c vector
typedef mVector<double,allocator<double> > dv_;  // shorthand c++ vector
typedef struct c_double_matrix cdm_;            // short hand double c matrix
typedef mMatrix<double,allocator<double> > dm_;  // shorthand c++ matrix

typedef struct c_complex_vector ccv_;            // short hand complex c vector
typedef mVector<c_complex,allocator<c_complex> > cv_;  // shorthand c++ vector
typedef struct c_complex_matrix ccm_;            // short hand complex c matrix
typedef mMatrix<c_complex,allocator<c_complex> > cm_;  // shorthand c++ matrix

// Some explicit instantiations to work around bug in g++ templates

template class mVector<c_complex,mu_allocator<allocator<c_complex>,rc_allocator<allocator<c_complex> > > >;

template class mVector<c_complex,mu_allocator<rc_allocator<allocator<c_complex> >,allocator<c_complex> > >;

template class mVector<c_complex,mu_allocator<rc_allocator<allocator<c_complex> >,rc_allocator<allocator<c_complex> > > >;

template class mVector<c_complex,mu_allocator<scalar_allocator<c_complex>,rc_allocator<allocator<c_complex> > > >;

/* element access functions */

extern "C"
long unsigned
a_elem(long unsigned s)
{
  return s;
}

extern "C"
long unsigned
v_elem(long unsigned s)
{
  return s-1;
}

extern "C"
long unsigned
m_elem(long unsigned rows,long unsigned cols,long unsigned r,long unsigned c)
{
  return (r-1)*cols+(c-1);
}

void
convert(cdv_& a,dv_& b) 
{
  b.resize(a.s_);
  b.container()=a.data_-1;
}
  
void
convert(cdm_& a,dm_& b) 
{
  b.resize(a.r_,a.c_);
  b.container()=a.data_-1;
}

void
convert(ccv_& a,cv_& b) 
{
  b.resize(a.s_);
  b.container()=a.data_-1;
}
  
void
convert(ccm_& a,cm_& b)
{
  b.resize(a.r_,a.c_);
  b.container()=a.data_-1;
}

extern "C"
void 
complex_gauss_jordan(ccm_ opm,ccm_ a,double tol,long unsigned full,long unsigned r,long unsigned c)
{
  cm_ opmp;
  convert(opm,opmp);
  cm_ ap;
  convert(a,ap);
  gauss_jordan(opmp,ap,tol,full,r,c);
}

extern "C"
int 
complex_lud(ccm_ opm,long unsigned* swp,double tol)
{
  cm_ opmp;
  convert(opm,opmp);
  mVector<size_t,allocator<size_t> > swpp(opm.r_);     // make a c++ vector
  swpp.container()=swp;                  // convert to reference seantics to swp
  lud(opmp,swpp,tol);
}

extern "C"
void 
complex_lu_solve(ccm_ opm,ccm_ a,double tol) 
{
  cm_ opmp;
  convert(opm,opmp);
  cm_ ap;
  convert(a,ap);
  lu_solve(opmp,ap,tol);
}

extern "C"
c_complex 
complex_determinant(ccm_ opm,double tol) 
{
  cm_ opmp;
  convert(opm,opmp);
  return determinant(opmp,tol);
}

extern "C"
void
complex_qrd(ccm_ opm,ccm_ q)
{
  cm_ opmp;
  convert(opm,opmp);
  cm_ qp;
  convert(q,qp);
  qrd(opmp,qp);
}

extern "C"
void 
complex_hessenberg(ccm_ opm,long unsigned record)
{
  cm_ opmp;
  convert(opm,opmp);
  hessenberg(opmp,record);
}

extern "C"
void 
complex_bidiag(ccm_ opm,long unsigned record)
{
  cm_ opmp;
  convert(opm,opmp);
  bidiag(opmp,record);
}

extern "C"
void
complex_eigenvalues(ccm_ opm,double tol)
{
  cm_ opmp;
  convert(opm,opmp);
  eigenvalues(opmp,tol);
}

extern "C"
void
complex_svd(ccm_ opm,ccm_ u,cdv_ d,ccm_ v,double tol)
{
  cm_ opmp;
  convert(opm,opmp);
  cm_ up;
  convert(u,up);
  cm_ vp;
  convert(v,vp);
  dv_ dp;
  convert(d,dp);
  svd(opmp,up,dp,vp,tol);
}

extern "C"
void
complex_singular_values(ccm_ opm,cdv_ sv,double tol)
{
  cm_ opmp;
  convert(opm,opmp);
  dv_ svp;
  convert(sv,svp);
  singular_values(opmp,svp,tol);
}

extern "C"
long unsigned
complex_pseudo_inverse(ccm_ opm,ccm_ u,cdv_ sv,ccm_ v,double tol)
{
  cm_ opmp;
  convert(opm,opmp);
  cm_ up;
  convert(u,up);
  cm_ vp;
  convert(v,vp);
  dv_ svp;
  convert(sv,svp);
  pseudo_inverse(opmp,up,svp,vp,tol);
}

extern "C"
void
complex_mdotm(ccm_ a,ccm_ b,ccm_ r)
{
  cm_ ap;
  convert(a,ap);
  cm_ bp;
  convert(b,bp);
  cm_ rp;
  convert(r,rp);
  dot(ap,bp,rp);
}

extern "C"
void
complex_mdotv(ccm_ a,ccv_ b,ccv_ r)
{
  cm_ ap;
  convert(a,ap);
  cv_ bp;
  convert(b,bp);
  cv_ rp;
  convert(r,rp);
  dot(ap,bp,rp);
}

extern "C"
void
complex_vdotm(ccv_ a,ccm_ b,ccv_ r)
{
  cv_ ap;
  convert(a,ap);
  cm_ bp;
  convert(b,bp);
  cv_ rp;
  convert(r,rp);
  dot(ap,bp,rp);
}

  
