// tests the linear algebra functions based on Householder reflections
// and Givens rotations

/*
Copyright (C) 1996 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
You should have received a copy of the GNU General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include<complex.h>
#include<mMatrix.h>
#include<instance.h>
#include<lin_alg.h>

int main(int argc,char **argv) {
  int dim=atoi(argv[1]);
  mMatrix<double_complex,allocator<double_complex> > a(dim,dim),i(dim,dim);
  mVector<double_complex,allocator<double_complex> > p(dim),q(dim),z(dim);
  rand(1);              // seed the random # generator with a known value
  double maxr=(double)0x0fffffff;
  for(int r=1;r<=dim;r++) {  // set a to a random matrix, i to the identity
    for(int c=1;c<=dim;c++) {
      a(r,c)=double_complex(rand()/maxr,rand()/maxr);
      i(r,c)=0.0;
    }
    i(r,r)=1.0;
    p(r)=double_complex(rand()/maxr,rand()/maxr);   // and p to a random vector
  }
  mMatrix<double_complex,allocator<double_complex> > b(dim,dim),c(dim,dim);
  b=a;        // copy the random matrix and the identity
  c=i;
  cout << "Test mMatrix is random complex, with "<< dim << " rows and columns\n";
  mMatrix<double_complex,allocator<double_complex> > u(dim,dim),v(dim,dim);
  u=i;                // identity matrices
  v=i;
  mVector<double,allocator<double> > sv1(dim);

  cout << "Test QR decomposition, using the sup norm with abs: \n";
  qrd(b,v);
  dot(v,b,u);
  cout << "Reconstruction, deviation from original: ";
  cout << sup(u-a) << "\n";

  cout << "Test singular value decomposition, using the sup norm with abs: \n";
  b=a;
  u=i;                // identity matrices
  v=i;
  svd(b,u,sv1,v,0.0);
  cout << "Ratio, smallest to largest SV: " << sv1(dim)/sv1(1) << '\n';
  v.transpose();       // Hermitian adjoint of v
  double_complex (*cp)(const double_complex&)=conj;
  transform(v,v,cp);
  for(int r=1;r<=dim;r++) v.row(r)*=(double_complex)sv1(r);
  cout << "Reconstruction, deviation from original: ";
  cout << sup(a-dot(u,v,b)) << '\n';

  cout << "Nonsingular inverse by pseudo_inverse():\n";
  b=a;         // reset b, u and v
  u=v=i;
  pseudo_inverse(b,u,sv1,v,0.0);
  cout << "Deviation from the inverse: " << sup(i-dot(b,a,c))<< '\n';

  cout << "Singular inverse by pseudo_inverse():\n";
  b=a;         // reset b, u and v
  u=v=i;
  b.col(dim)=b.col(1);      // set one column equal to another
  c=b;              // copy it into a temporary
  pseudo_inverse(b,u,sv1,v,0.0);    // invert and test
//  q=u.col(dim);     // the left singular vector
assign(u.col(dim),q);
  transform(q,q,cp);    // adjoint
  cout << "Test singular vector: " << sup(dot(q,c,z)) << "\n";
  double (*np)(const double_complex&)=norm;
  cout << "Normalization: " << 1.0-accumulate(q,0.0,np) << "\n";
  double_complex ponq=sum(q*p);
  transform(q,q,cp);    // adjoint
  p-=(ponq*q);       // subtract off the singular part of the random vector
  dot(b,p,q);             // the solution to the inverse problem
  cout << "Deviation of solution: " << sup(p-dot(c,q,z)) << '\n';

  cout << "Eigenvalues and eigenvectors:\n";
  b=a;         // reset b
  eigenvalues(b,0.0);     // calculate all eigenvalues
//  q=b.diagonal();
assign(b.diagonal(),q);
  for(int r=1;r<=dim;r++) {
//    b=a-(q(r)*i);       // subtract off the eigenvalue
assign(a-(q(r)*i),b);       // subtract off the eigenvalue
    u=v=i;      // reset u and v to the identity
    svd(b,u,sv1,v,0.0);
      // The columns of u and v are now the left and right singular vectors
      // of b.  The last columns are the left and right eigenvectors.
    cout << "Deviation of eigenpair " << r << ": ";
    cout << sup(dot(a,v.col(dim),z) - q(r)*v.col(dim)) << '\n';
  }

  cout << "The accuracy of these routines is indicated by the size of the ";
  cout << "numbers reported above.\n";

  return 0;
}
