// This file is a modification of c++ complex header files included in the 
// libg++ distribution.  The copyright from those files in included in 
// its entirety below and applies to this modified code as well.
// Modifications began on December 22, 1995 by Robert D. Pierce.

// ORIGINAL COPYRIGHT STATEMENT:

// The template and inlines for the -*- C++ -*- complex number classes.
// Copyright (C) 1994 Free Software Foundation

// This file is part of the GNU ANSI C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the terms of
// the GNU General Public License as published by the Free Software
// Foundation; either version 2, or (at your option) any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// As a special exception, if you link this library with files compiled
// with a GNU compiler to produce an executable, this does not cause the
// resulting executable to be covered by the GNU General Public License.
// This exception does not however invalidate any other reasons why the
// executable file might be covered by the GNU General Public License.

// Written by Jason Merrill based upon the specification in the 27 May 1994
// C++ working paper, ANSI document X3J16/94-0098.

#include <math.h>
#include <complex.h>
#include <iostream.h>

extern "C"
struct c_complex {
  double re;
  double im;
  c_complex() : re(0.0), im(0.0) {}
  c_complex(const c_complex& a) : re(a.re), im(a.im) {}
  c_complex(double r,double i) : re(r), im(i) {}
  double real() { return re; }
  double imag() { return im; }

  c_complex& operator = (const c_complex& r) { 
    re = r.re; im = r.im; return (*this); }

  c_complex& operator = (const complex<double>& r) {
    re = real(r); im = imag(r); return (*this); }

};

c_complex&
operator += (c_complex& a,const c_complex& r)
{
  a.re += r.re;
  a.im += r.im;
  return a;
}

c_complex&
operator -= (c_complex& a,const c_complex& r)
{
  a.re -= r.re;
  a.im -= r.im;
  return a;
}

c_complex&
operator *= (c_complex& a,const c_complex& r)
{
  double f = a.re * r.re - a.im * r.im;
  a.im = a.re * r.im + a.im * r.re;
  a.re = f;
  return a;
}

double
imag (const c_complex& x)
{
  return x.im;
}

double
real (const c_complex& x)
{
  return x.re;
}

c_complex
operator + (const c_complex& x, const c_complex& y)
{
  return c_complex (real (x) + real (y), imag (x) + imag (y));
}

c_complex
operator + (const c_complex& x, double y)
{
  return c_complex (real (x) + y, imag (x));
}

c_complex
operator + (double x, const c_complex& y)
{
  return c_complex (x + real (y), imag (y));
}

c_complex
operator - (const c_complex& x, const c_complex& y)
{
  return c_complex (real (x) - real (y), imag (x) - imag (y));
}

c_complex
operator - (const c_complex& x, double y)
{
  return c_complex (real (x) - y, imag (x));
}

c_complex
operator - (double x, const c_complex& y)
{
  return c_complex (x - real (y), - imag (y));
}

c_complex
operator * (const c_complex& x, const c_complex& y)
{
  return c_complex (real (x) * real (y) - imag (x) * imag (y),
			   real (x) * imag (y) + imag (x) * real (y));
}

c_complex
operator * (const c_complex& x, double y)
{
  return c_complex (real (x) * y, imag (x) * y);
}

c_complex
operator * (double x, const c_complex& y)
{
  return c_complex (x * real (y), x * imag (y));
}

c_complex
operator / (const c_complex& x, double y)
{
  return c_complex (real (x) / y, imag (x) / y);
}

c_complex
operator + (const c_complex& x)
{
  return x;
}

c_complex
operator - (const c_complex& x)
{
  return c_complex (-real (x), -imag (x));
}

bool
operator == (const c_complex& x, const c_complex& y)
{
  return real (x) == real (y) && imag (x) == imag (y);
}

bool
operator == (const c_complex& x, double y)
{
  return real (x) == y && imag (x) == 0;
}

bool
operator == (double x, const c_complex& y)
{
  return x == real (y) && imag (y) == 0;
}

bool
operator != (const c_complex& x, const c_complex& y)
{
  return real (x) != real (y) || imag (x) != imag (y);
}

bool
operator != (const c_complex& x, double y)
{
  return real (x) != y || imag (x) != 0;
}

bool
operator != (double x, const c_complex& y)
{
  return x != real (y) || imag (y) != 0;
}

double
norm (const c_complex& x)
{
  return real (x) * real (x) + imag (x) * imag (x);
}

double
abs (const c_complex& x)
{
  return sqrt (norm(x));
}

double
arg (const c_complex& x)
{
  return atan2 (imag (x), real (x));
}

c_complex
conj (const c_complex& x) 
{
  return c_complex (real (x), -imag (x));
}

c_complex
cos (const c_complex& x)
{
  return c_complex (cos (real (x)) * cosh (imag (x)),
			   - sin (real (x)) * sinh (imag (x)));
}

c_complex
cosh (const c_complex& x)
{
  return c_complex (cosh (real (x)) * cos (imag (x)),
			   sinh (real (x)) * sin (imag (x)));
}

c_complex
c_polar (double r, double t)
{
  return c_complex (r * cos (t), r * sin (t));
}

c_complex
exp (const c_complex& x)
{
  return c_polar (double (exp (real (x))), imag (x));
}

c_complex
log (const c_complex& x)
{
  return c_complex (log (abs (x)), arg (x));
}

c_complex
pow (const c_complex& x, const c_complex& y)
{
  double logr = log (abs (x));
  double t = arg (x);

  return c_polar (double (exp (logr * real (y) - imag (y) * t)),
		double (imag (y) * logr + real (y) * t));
}

c_complex
pow (const c_complex& x, double y)
{
  return exp (double (y) * log (x));
}

c_complex
pow (double x, const c_complex& y)
{
  return exp (y * double (log (x)));
}

c_complex
sin (const c_complex& x)
{
  return c_complex (sin (real (x)) * cosh (imag (x)),
			   cos (real (x)) * sinh (imag (x)));
}

c_complex
sinh (const c_complex& x)
{
  return c_complex (sinh (real (x)) * cos (imag (x)),
			   cosh (real (x)) * sin (imag (x)));
}

istream&
operator >> (istream& is, c_complex& x)
{
  double re, im = 0;
  char ch = 0;

  if (is.ipfx0 ())
    {
      if (is.peek () == '(')
	is >> ch;
      is >> re;
      if (ch == '(')
	{
	  is >> ch;
	  if (ch == ',')
	    is >> im >> ch;
	}
    }
  is.isfx ();

  if (ch != 0 && ch != ')')
    is.setstate (ios::failbit);
  else if (is.good ())
    x = c_complex (re, im);

  return is;
}

ostream&
operator << (ostream& os, const c_complex& x)
{
  return os << '(' << real (x) << ',' << imag (x) << ')';
}

// The code below is adapted from f2c's libF77, and is subject to this
// copyright:

/****************************************************************
Copyright 1990, 1991, 1992, 1993 by AT&T Bell Laboratories and Bellcore.

Permission to use, copy, modify, and distribute this software
and its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the names of AT&T Bell Laboratories or
Bellcore or any of their entities not be used in advertising or
publicity pertaining to distribution of the software without
specific, written prior permission.

AT&T and Bellcore disclaim all warranties with regard to this
software, including all implied warranties of merchantability
and fitness.  In no event shall AT&T or Bellcore be liable for
any special, indirect or consequential damages or any damages
whatsoever resulting from loss of use, data or profits, whether
in an action of contract, negligence or other tortious action,
arising out of or in connection with the use or performance of
this software.
****************************************************************/

c_complex& 
operator /= (c_complex& a,const c_complex& y)
{
  double ar = abs (y.re);
  double ai = abs (y.im);
  double nr, ni;
  double t, d;
  if (ar <= ai)
    {
      t = y.re / y.im;
      d = y.im * (1 + t*t);
      nr = (a.re * t + a.im) / d;
      ni = (a.im * t - a.re) / d;
    }
  else
    {
      t = y.im / y.re;
      d = y.re * (1 + t*t);
      nr = (a.re + a.im * t) / d;
      ni = (a.im - a.re * t) / d;
    }
  a.re = nr;
  a.im = ni;
  return a;
}

c_complex
operator / (const c_complex& x, const c_complex& y)
{
  double ar = abs (real (y));
  double ai = abs (imag (y));
  double nr, ni;
  double t, d;
  if (ar <= ai)
    {
      t = real (y) / imag (y);
      d = imag (y) * (1 + t*t);
      nr = (real (x) * t + imag (x)) / d;
      ni = (imag (x) * t - real (x)) / d;
    }
  else
    {
      t = imag (y) / real (y);
      d = real (y) * (1 + t*t);
      nr = (real (x) + imag (x) * t) / d;
      ni = (imag (x) - real (x) * t) / d;
    }
  return c_complex (nr, ni);
}

c_complex
operator / (double x, const c_complex& y)
{
  double ar = abs (real (y));
  double ai = abs (imag (y));
  double nr, ni;
  double t, d;
  if (ar <= ai)
    {
      t = real (y) / imag (y);
      d = imag (y) * (1 + t*t);
      nr = x * t / d;
      ni = -x / d;
    }
  else
    {
      t = imag (y) / real (y);
      d = real (y) * (1 + t*t);
      nr = x / d;
      ni = -x * t / d;
    }
  return c_complex (nr, ni);
}

c_complex
pow (const c_complex& xin, int y)
{
  if (y == 0)
    return c_complex (1.0);
  c_complex r (1.0);
  c_complex x (xin);
  if (y < 0)
    {
      y = -y;
      x = 1/x;
    }
  for (;;)
    {
      if (y & 1)
	r *= x;
      if (y >>= 1)
	x *= x;
      else
	return r;
    }
}

c_complex
sqrt (const c_complex& x)
{
  double r = abs (x);
  double nr, ni;
  if (r == 0.0)
    nr = ni = r;
  else if (real (x) > 0)
    {
      nr = sqrt (0.5 * (r + real (x)));
      ni = imag (x) / nr / 2;
    }
  else
    {
      ni = sqrt (0.5 * (r - real (x)));
      if (imag (x) < 0)
	ni = - ni;
      nr = imag (x) / ni / 2;
    }
  return c_complex (nr, ni); 
}

