/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 * Name          : pnm2nc.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : 7 Sep 1994
 * Modified on   : 
 * Synopsis      :
 *
 * Usage:
 *  pnm2nc [-h] ncName [-v varName] [-t dataType] [name1 name2 ...]
 *
 * Where:
 *   -h         Prints this help text.
 *   ncName     Name of netCDF file to process, must be first parameter.
 *   varName    Name of variable to create for the image(sequence).
 *              Default is 'name1'. If the variable allready exists, then
 *              the images are appended to that variable.
 *   dataType   Type of data to generate in the variable. Possible values:
 *                byte long
 *   name1 ...  Names of the imagefiles to read.
 *              Recognized formats:
 *                pbm ascii, pgm ascii, ppm ascii, pgm raw, ppm raw.
 *              If no name is given, stdin is read.
 *
 * Description:
 *   pnm2nc opens ncName or creates it, if non-existent and copies the
 *   specified images in the variable 'varName'. ppm is stored as NC_LONG.
 *   All other data as NC_BYTE, unless specified different with -t. 
 */

#include "pips.h"   /* PIPS system header file  */

/*
 * Global definitions
 */
/*
 * Description of program for display on stderr by pipsPrintError()
 */
char *programName = "pnm2nc";
char *description[] = {
  "  pnm2nc [-h] ncName [-v varName] [-t dataType] [name1 name2 ...] ",
  " ",
  "Where:",
  "  -h         Prints this help text.",
  "  ncName     Name of netCDF file to process, must be first parameter.",
  "  varName    Name of variable to create for the image(sequence).",
  "             Default is 'name1'. If the variable allready exists, then",
  "             the images are appended to that variable.",
  "  dataType   Type of data to generate in the variable. Possible values:",
  "               byte long",
  "  name1 ...  Names of the imagefiles to read.",
  "             Recognized formats:",
  "               pbm ascii, pgm ascii, ppm ascii, pgm raw, ppm raw.",
  "             If no name is given, stdin is read.",
  " ",
  "Description:",
  "  pnm2nc opens ncName or creates it, if non-existent and copies the",
  "  specified images in the variable 'varName'. ppm is stored as NC_LONG.",
  "  All other data as NC_BYTE, unless specified different with -t. ",
  ""
}; /* end description */


/******************************************************************************
 *   M A I N
 ******************************************************************************
 */
#ifdef __STDC__

int main (int argc, char *argv[])

#else

main (argc, argv)
    int argc;
    char *argv[];

#endif
{
  char varName [MAX_NC_NAME];  /* Name of variable to create                 */
  nc_type dataType = NC_UNSPECIFIED;     /* Data type to use for variable    */
  long i;                      /* index variable                             */
  int histIndex;               /* Variable for constructing history string   */
  int  ncId        = -1;       /* netcdf file Id                             */
  char *ncName     = NULL;     /* Name of netcdf file                        */
  char *position   = NULL;     /* Used to 'protect' strlen() from a NULL arg */
  int  readStdin   = 1;        /* set to 0, if image file names found        */
  char *history    = NULL;     /* Saved command line parameters              */

/******************************************************************************
 *   A N A L Y Z E   C O M M A N D   L I N E   P A R A M E T E R S
 ******************************************************************************
 */
/*
 * If no parameters given, complain about missing netcdf filename
 */
  if (argc < 2)
    pipsPrintError (-1, PipsNetcdfMissing, NULL, PipsExit);

/*
 * If the -h option is given, print explanation and quit
 */
  if (strncmp ("-h", argv[1], 2) == 0)
    pipsPrintError (-1, PipsOk, NULL, PipsExit);

/*
 * First argument is expected to be the name of the netcdf file
 */
  ncName = argv[1];

/*
 * Examine argv[i]: look for parameters '-v', '-t'
 */
  varName [0] = 0;
  varName [MAX_NC_NAME-1] = 0;
  i = 2;
  while ((i < argc) && (argv[i][0] == '-')) {
    if (i+1 >= argc) 
      pipsPrintError (-1, PipsSinglePara, argv[i], PipsExit);
    else
      switch (argv[i][1]) {
/*
 * '-v' option : store next parameter in 'varName'
 */
      case 'v' : strncpy (varName, argv[++i], MAX_NC_NAME-1);
                 break;
/*
 * '-t' option : store next parameter in 'dataType'
 */
      case 't' : position = strstr ("bytelong", argv[++i]);
                 switch (strlen (position)) {
                 case 8 : dataType = NC_BYTE;   break;
                 case 4 : dataType = NC_LONG;   break;
                 default :
                   pipsPrintError (-1,PipsBadParameter, argv[i], PipsExit);
                 break;
                 } /* end switch */
                 break;
      default  : pipsPrintError (-1, PipsBadParameter, argv[i], PipsExit);
      } /* end switch */
    i++;
  } /* end while */


/*
 * If no further parameters are given, read from stdin
 */
  readStdin = (argc == i);

/******************************************************************************
 *  O P E N   N E T C D F   F I L E ,   G E T   I M A G E   D I M E N S I O N
 ******************************************************************************
 */
/*
 * Prevent error messages (error codes are evaluated for further calls)
 */
  ncopts = NC_NOERR;

/*
 * Open an EXISTING netCDF file 
 */
  ncId = ncopen (ncName, NC_WRITE);

/*
 * Attempt to create a new file. Abort program on failure
 * Otherwise go in define mode for adding history attribute
 */
  if (ncId == -1) {          /* netcdf file doesn't exist         */
    ncId = nccreate (ncName, NC_CLOBBER);
    if (ncId == -1)
      pipsPrintError (-1, PipsErrorFileWrite, ncName, PipsExit);
  } else
    ncredef (ncId);

/*
 * Save the command line parameters in the history attribute (concatenated)
 */
  for (histIndex = 0; histIndex < argc; histIndex++) {
    pipsStrApp (&history, argv[histIndex]);
    pipsStrApp (&history, " ");
  } /* end for */
  pipsHistory (ncId, history);
  free (history);
  ncendef (ncId);

/*
 * Prevent error messages (error codes are evaluated for further calls)
 */
  ncopts = NC_NOERR;

/******************************************************************************
 *   A D D   I M A G E (S)   T O   T H E   N E T C D F   F I L E
 ******************************************************************************
 */
/*
 * Check now for input file names
 */
  for (; i < argc; i++) {
    pipsAddPnm (ncId, varName, argv[i], dataType);
  }  /* endfor */

/*
 * Read images from stdin, if no image filename was given
 * until EOF is encountered
 */
  if (readStdin)
    while (pipsAddPnm (ncId, varName, NULL, dataType) == PipsOk)
      ; /* continue reading */ 

/*
 * Commit changes and close netCDF-file
 */
  if (ncclose (ncId) == -1)
    pipsPrintError (-1, PipsErrorFileWrite, ncName, PipsExit);
  return (PipsOk);
} /* end main */


/* end pnm2nc.c */
