/*
This file is part of GNU PIPS,
    version 1.01: Parallel Information Processing System

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 *        PIPS version 1.01:  Parallel Information Processing System
 *         Imperial College of Science, Technology and Medicine
 *                      Author: Frank Heimes
 */
/*
 *	pips.h
 *
 *	Libpips.a symbols and includes.
 *
$Log$
 */

#ifndef	_PIPS_H_
#define	_PIPS_H_


/******************************************************************************
 *   I N C L U D E   F I L E S 
 ******************************************************************************
 */

#include <stdio.h>        /* Standard C I/O library                          */
#include <stdlib.h>
#include <strings.h>
#include <float.h>
#include <limits.h>
#include <time.h>         /* Standard C time handling routines               */
#include "netcdf.h"       /* NetCDF version 2.3.2 header file                */


/******************************************************************************
 *   S Y M B O L S
 ******************************************************************************
 */

/*
 * Image data types currently recognized by pips functions
 */
#define PipsUnknown   0
#define PipsPbmAscii  1
#define PipsPgmAscii  2
#define PipsPpmAscii  3
#define PipsPbmRaw    4
#define PipsPgmRaw    5
#define PipsPpmRaw    6

/*
 * Error codes, used for pipsPrintError()
 */
#define PipsOk             0 /* No errors                                    */
#define PipsErrorFileRead  1 /* Could not open existing file for reading     */
#define PipsErrorFileWrite 2 /* Could not open/create file                   */
#define PipsErrorFileExist 3 /* An existing file would be overwritten        */
#define PipsNetcdfMissing  4 /* Filename for netCDF file is missing          */
#define PipsBadParameter   5 /* Unknown option found                         */
#define PipsSinglePara     6 /* Option that expects parameter, was single    */
#define PipsNoOption       7 /* Not enough options on command line           */
#define PipsNoMemory       8 /* Insufficient memory for operation            */
#define PipsImageType      9 /* The image type was not recognized            */
#define PipsInvalidVar    10 /* Invalid variable name specified              */
#define PipsDataIncompat  11 /* Variable data is not compatible to pnm data  */
#define PipsPrematureEof  12 /* Error in att. def.: EOF before end of def.   */
#define PipsVarOrColonExp 13 /* Error in att. def.: Var. or Colon expected   */
#define PipsColonExpected 14 /* Error in att. def.: Colon expected           */
#define PipsAttNameExp    15 /* Error in att. def.: Attribute name expected  */
#define PipsEqualSignExp  16 /* Error in att. def.: '=' expected             */
#define PipsAttValueExp   17 /* Error in att. def.: Attribute value(s) exp.  */
#define PipsSemikolonExp  18 /* Error in att. def.: Semikolon expected       */
#define PipsVariableType  19 /* Incompatible variable type encountered       */
#define PipsPvmInactive   20 /* Task could not connect to the pvm-deamon     */
#define PipsPrivatePara   21 /* Forbidden parameter used as command line opt */

/*
 * Return values (that don't have an associated error message string)
 */
#define PipsNoImages 99      /* Stdin returned EOF when reading images       */

/*
 * Flags for 'exit' in pipsPrintError ()
 */
#define PipsDontExit  0      /* Print error message but don't exit           */
#define PipsExit      1      /* Print error message, close netcdf file, exit */

/*
 * Message Tags for signalling pipsJobManager() <-> pipsProcPar()
 */
#define PipsWaiting   1      /* pipsProcPar is waiting for data to process   */
#define PipsCommence  2      /* pipsProcPar received data and should start   */

/*
 * Accept a message from any task
 */
#define PipsAnyTask  -1

/*
 * Command line argument to identify a child; passed to spawned tasks
 */
#define PipsChildTag  "PipsYouAreAChild" 

/*
 * Name used as attribute for the read/written image file type
 */
#define PipsFileTypeAtt "identifier"

/*
 * Name used as history attribute prefix
 */
#define PipsHistPrefix "__history_"

/*
 * Job id that is used for 'idle task; not doing any job'
 */
#define PipsNoJob INT_MAX  

/*
 * Version number for printout in pipsPrintError()
 */
#define PipsVersion "1.01"


/******************************************************************************
 *   M A C R O S
 ******************************************************************************
 */
#define PipsMax(X,Y)    ((X) > (Y)) ? (X) : (Y) 
#define PipsMin(X,Y)    ((X) < (Y)) ? (X) : (Y) 

/*
 * Macros to avoid 'differ in signedness' warning
 */
#define pvm_pkubyte(X,Y,Z)   pvm_pkbyte   ((char  *)(X), (Y), (Z))
#define pvm_upkubyte(X,Y,Z)  pvm_upkbyte  ((char  *)(X), (Y), (Z))
#define pvm_upkuint(X,Y,Z)   pvm_upkint   ((signed int   *)(X), (Y), (Z))
#define pvm_upkulong(X,Y,Z)  pvm_upklong  ((signed long  *)(X), (Y), (Z))
#define pvm_upkushort(X,Y,Z) pvm_upkshort ((signed short *)(X), (Y), (Z))


/******************************************************************************
 *   F U N C T I O N   P R O T O T Y P E S 
 ******************************************************************************
 */

#ifdef __cplusplus
extern "C" {
#endif

#ifdef __STDC__

int  pipsAddPnm           (int ncId, char *varName, char *imgName,
                           nc_type dataType);
void pipsAwaitChildData   (int tid[], int jobId[], int perform[], int *done,
                           int ntask, int *finishedJob, int *index);
int  pipsChangeAtts       (int ncId, char *defName);

long pipsDataPerJob       (int ntask, int granularity, int lower, int upper,
                           long items, int *njobs);
int  pipsExtPnm           (int ncId, char *varName, long number, char *imgName,
                           int imgType);
void pipsHistory          (int ncID, char *value);

int  pipsJobAvailable     (int jobId[], int *job, int ntask, int task,
                           int newJobs);
void pipsJobManager       (int ntask, int tid[], int ncId, int parac,
                           char **parav);
int  pipsMinMax           (int ncId, int varId, long number, long height,
                           long width, unsigned long *min, unsigned long *max);
int *pipsNewFlagArray     (int size, int initValue);

void pipsPrintError       (int ncId, int result, char *cause, int action);

void pipsPrintPerformance (int tid[], int perform[], int ntask, int njobs);

void pipsProcPar          (void);

int  pipsReadAttDef       (FILE *file, char *varName, char *attName,
                           nc_type *attType, int *attNum, char *attValueArray);
int  pipsReadPnmHeader    (FILE *file,
                           long *width, long *height);
int  pipsReadPnmValue     (FILE *file, int imgType, char *data);

void pipsSelectNewJob     (int *newJobs, int njobs, int *job);

int  pipsStrApp           (char **s1, char *s2);

void pipsTaskManager      (char *prgName, int *ntask, int **tid);

int  pipsVarDef           (int ncId, char *varName, nc_type *dataType,
                           int *fileType, long *image, long *height,
                           long *width);
int  pipsVarInq           (int ncId, int *dataId, char *varName,
                           nc_type *dataType, long *number, long *height,
                           long *width);
int  pipsWritePnmHeader   (FILE *file, int imgType, long width, long height);

int  pipsWritePnmValue    (FILE *file, int imgType, char *data);

#else

pipsAddPnm           ();
pipsAwaitChildData   ();
pipsChangeAtts       ();
pipsDataPerJob       ();
pipsExtPnm           ();
pipsExtPnms          ();
pipsHistory          ();
pipsJobAvailable     ();
pipsJobManager       ();
pipsNewFlagArray     ();
pipsMinMax           ();
pipsPrintError       ();
pipsPrintPerformance ();
pipsProcPar          ();
pipsReadAttDef       ();
pipsReadPnmHeader    ();
pipsReadPnmValue     ();
pipsSelectNewJob     ();
pipsStrapp           ();
pipsTaskManager      ();
pipsVarDef           ();
pipsVarInq           ();
pipsWritePnmHeader   ();
pipsWritePnmValue    ();

#endif

#ifdef __cplusplus
}
#endif

#endif	/*_PIPS_H_*/
