/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 * Name          : pips/src/pipsAwaitChildData.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 * Synopsis      : 
 *
 *   pipsAwaitChildData (tid, jobId, perform, done, ntask, finishedJob, index)
 *
 *   int tid []       IS   An array with task Id's of the tasks
 *   int jobId []     IS   An array with currently processed data fragments
 *   int perform []   IS   An array with number of completed jobs per task
 *   int *done        IS   Total number of accomplished jobs so far
 *   int ntask        IS   Number of spawned tasks
 *   int *finishedJob IS   Number of recently finished job.
 *   int *index       IS   Index for tid[], jobId[] and perform[]
 *
 * Description   : 
 *
 *   pipsAddPnm() waits for the next task to send an idle signal.
 *   It then looks up its task id and prepares the reception of data,
 *   if the task was busy with a job.
 */

#include "pips.h"   /* PIPS system header file  */
#include "pvm3.h"   /* PVM system header file   */

#ifdef __STDC__

void pipsAwaitChildData (int tid[], int jobId[], int perform[], int *done,
                          int ntask, int *finishedJob, int *index)

#else

pipsAwaitChildData (tid, jobId, perform, done, ntask, finishedJob, index)
    int tid[];
    int jobId[];
    int perform[];
    int *done;
    int ntask;
    int *finishedJob;
    int *index;

#endif
{
  int bufId;            /* Id of current buffer; used to determine sender    */
  int taskId;           /* TaskId of task that accomplished a job            */
  long i;               /* Index to jobId[]                                  */

/*
 * Wait for any child to send PipsWaiting and then look up its taskId 
 * in tid[]. The found index will address tid[], jobId[] and perform[]
 */
  bufId = pvm_recv (PipsAnyTask, PipsWaiting);

/*
 * Get sender's task Id
 */
  pvm_bufinfo (bufId, NULL, NULL, &taskId);

/*
 * Look up the index associated with it
 */
  for (*index=0; tid[*index] != taskId; (*index)++) ;

/*
 * If child was busy with a job, receive results and store them away
 */
  *finishedJob = jobId [*index];
  if (*finishedJob != PipsNoJob) {
    (*done)++;                  /* Increase total number of finished jobs  */
    (perform [*index])++;       /* Number of jobs finished by current task */
/*
 * Clear all entries of the finished job in the list of curr. processed jobs
 */
    for (i=0; i < ntask; i++)
      if (jobId [i] == *finishedJob)
        jobId [i] = PipsNoJob;
  }  /* end if *finishedJob   */
}  /* pipsAwaitChildData */


/* end pipsAwaitChildData.c */
