/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 * Name          : pips/src/pipsChangeAtts.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 * Synopsis      : 
 *
 *   pipsChangeAtts (ncId, defName)
 *
 *   int ncId            IS  Id of a netcdf file, whose attr. are to change
 *   char *defName       IS  Name of file containing attribute definitions
 * 
 * Description   : 
 *
 *   Opens the file 'defName' and changes/adds/deletes attributes in the
 *   file of 'ncId'. The definitions have to be in CDL-Syntax
 */

#include "pips.h"   /* PIPS system header file  */

#ifdef __STDC__

int pipsChangeAtts (int ncId, char *defName)

#else

pipsChangeAtts (ncId, defName)
    int ncId;
    char *defName;

#endif
{
  int varId = -1;               /* Variables id, whose attr. is changed      */
  FILE *file;                   /* File handle of definitions file           */
  nc_type attType;              /* attribute type: NC_CHAR,NC_LONG,NC_DOUBLE */
  int attNum;                   /* number of attribute values in attr. defin.*/
  int result;                   /* result of a fscanf attempt                */
  int i;                        /* index variable                            */
  char varName [MAX_NC_NAME];   /* Name of variable                          */
  char attName [MAX_NC_NAME];   /* Name of attribute to change               */
  char attValue [MAX_NC_NAME*100];      /* List of attribute values */

/*
 * Attribute change only allowed in definition-mode
 */
  ncredef (ncId);

/*
 * Open definitions file and return error code on failure
 * empty name means: read stdin
 */
 if (! defName)
    file = stdin;
  else
    file = fopen (defName, "rb");
  if (!file)
    pipsPrintError (ncId, PipsErrorFileRead, defName, PipsExit);

/*
 * Read attribute definitions one by one from 'file'
 * and apply them to current netcdf file
 */
  do {

/*
 * Initialize attValue to 0
 */
    for (i=0; i < MAX_NC_NAME*100; i++)
      attValue [i] = 0;

/*
 * Get one attribute definition
 */
    result = pipsReadAttDef (file,varName,attName,&attType,&attNum,attValue);

/*
 * On success, apply definition on netcdf file
 */
    if (result == PipsOk) {

/*
 * Query variable id from netcdf file, if 'varName' is not empty
 * Ignore attribute definitions for nonexisting variables
 */
      varId = NC_GLOBAL;
      if (varName[0]) {
        varId = ncvarid (ncId, varName);
        if (varId == -1)
          continue;
      } /* end if */

/*
 * Delete attribute, if attribute value was empty
 */
      if (attNum == 0)
        ncattdel (ncId, varId, attName);
      else

/*
 * Change attribute values
 */
        ncattput (ncId,varId,attName,attType,attNum,(void *)attValue);
    } /* end if */
  } while (result != EOF);

/*
 * Close definitions file (if not stdin)
 */
  if (defName)
    fclose (file);

  return (PipsOk);
}  /* end pipsChangeAtts */


/* end pipsChangeAtts.c */
