/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU PIPS; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*/

/*
 * Name          : pips/src/pipsVarDef.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 * Synopsis      : 
 *
 *   pipsVarDef (ncId, varName, dataType, fileType, image, height, width)
 *
 *   int ncId          IS   the identifier returned by ncopen() or nccreate()
 *   char *varName     IS   name of variable to create, default is imgName
 *   nc_type *dataType IS   the datatype that is used in the variable
 *   int *fileType     IS   the type of image (to be) saved in the variable
 *   long *image       IS   the current number of images in 'varName' (if exist)
 *   long *height      IS   height of images, the variable should hold
 *   long *width       IS   width of images, the variable should hold
 * 
 * Description   : 
 *
 *   If the variable exists, then it returns it's properties in 'dataType',
 *   'fileType', 'image', 'height' and 'width', otherwise it
 *   creates the unlimited dimension and then 
 *   limited dimensions of size 'width' and 'height', if they don't exist.
 *   Creates a variable with name 'varName' and dimensions 'unlimited',
 *   'height', 'width', unless it exists. 'Image' is set to the current value
 *   of the unlimited dimension, so the number of images per variable. The 
 *   function leaves the define mode and returns the variable Id.
 */

#include "pips.h"   /* PIPS system header file  */

#ifdef __STDC__

int pipsVarDef (int ncId, char *varName, nc_type *dataType, int *fileType,
                 long *image, long *height, long *width)

#else

pipsVarDef (ncId, varName, dataType, fileType, image, height, width)
    int ncId;
    char *varName;
    nc_type *dataType;
    int *fileType;
    long *image;
    long *height;
    long *width;

#endif
{
/*
 * Variables for the netcdf file
 */
  static int dataId;      /* Data variable Id                                */
  int recDim;             /* record dimension == picture index               */
  long rowDim;            /* Image height dimension variable                 */
  long columnDim;	      /* Image width dimension variable                  */
  long size;              /* Size of an examined dimension                   */
  int n;			      /* Index for dimension parameters                  */
  int ndims;              /* Number of dimensions in the netcdf file         */
  int dims [MAX_VAR_DIMS];    /* Array with dimension ids for 'dataId'       */
  char name [20];         /* Name of new dimensions, appended by dim. size   */
  long fillValue;         /* Value for missing data: 0                       */
  char identifier[MAX_NC_NAME];  /* Identifier ('name' of fileType)          */

/*
 * Reenter define mode
 */
  ncredef (ncId);

/*
 * Prevent error messages (error codes are evaluated for further calls)
 */
    ncopts = NC_NOERR;

/******************************************************************************
 *  N O N E X I S T I N G   V A R I A B L E 
 ******************************************************************************
 */
  if ((dataId = ncvarid (ncId, varName)) == -1) {

/*
 * Query netcdf file: get record dimension
 */
    ncinquire (ncId, 0,0,0, &recDim);

/*
 * Define record dimension, if nonexistent
 * else store current number of images in 'image'
 */
    *image = 0;
    if (recDim == -1)
      recDim = ncdimdef (ncId, "recDim", NC_UNLIMITED);
    else
      ncdiminq (ncId, recDim, 0, image);

/*
 * Define new image dimensions for new image (-sequence)
 * or use allready existing dimensions
 */
    sprintf (name, "height_%ld", *height);
    ncinquire (ncId, &ndims, 0, 0, 0);

/*
 * Try to find an existing dimension with size 'height'
 * but do NOT select the unlimited dimension.
 */
    for (n=0, size= -1; (n < ndims) && (size != *height); n++)
      if (n != recDim)
        ncdiminq (ncId, n, 0, &size);

/*
 * Create a new dimension, if a suitable dimension doesn't exist
 */
    if (size == *height)
      rowDim = --n;
    else
      rowDim = ncdimdef (ncId, name, *height);

/*
 * Try to find an existing dimension with size 'width'
 * but do NOT select the unlimited dimension or the row dimension.
 * NOTE: ncvardef does not allow to use 1 dimension id for 2 dimensions
 */
    sprintf (name, "width_%ld", *width);
    for (n=0, size= -1; (n < ndims) && (size != *width); n++)
      if ((n != recDim) && (n != rowDim))
        ncdiminq (ncId, n, 0, &size);

/*
 * Create a new dimension, if a suitable doesn't exist
 */
    if (size == *width)
      columnDim = --n;
    else
      columnDim = ncdimdef (ncId, name, *width);

/*
 * Prepare dimensions for creation of 'varName'
 */
    n = 0;
    dims [n++] = recDim;
    dims [n++] = rowDim;
    dims [n++] = columnDim;

/*
 * Create variable 'varName', if it doesn't exist
 * and reset *image to 0 (since this variable doesn't contain images)
 */
    dataId = ncvardef (ncId, varName, *dataType, n, dims);
    *image = 0;

  } else {

/******************************************************************************
 *  E X I S T I N G   V A R I A B L E 
 ******************************************************************************
 */
    ncvarinq (ncId, dataId, NULL, dataType, &n, dims, NULL);
/*
 * Take the last three dimensions of the variable.
 * 'image', 'height' and 'width' are set to 1, if the appropriate
 * dimension doesn't exist.
 */
    (*image) = 1;  (*height) = 1;  (*width) = 1;
    if (n > 2)
      ncdiminq (ncId, dims[n-3], 0, image);
    if (n > 1)
      ncdiminq (ncId, dims[n-2], 0, height);   
    if (n > 0)
      ncdiminq (ncId, dims[n-1], 0, width);

  } /* end if ncvarid */

/*
 * Assign data attribute (the type of image the data came from)
 */
  identifier [0] = 0;              /* Make sure it contains an empty string */
  switch (*fileType) {
    case PipsPbmAscii : strcat (identifier, "pbm ascii");  break;
    case PipsPgmAscii : strcat (identifier, "pgm ascii");  break;
    case PipsPpmAscii : strcat (identifier, "ppm ascii");  break;
    case PipsPbmRaw   : strcat (identifier, "pbm raw");    break;
    case PipsPgmRaw   : strcat (identifier, "pgm raw");    break;
    case PipsPpmRaw   : strcat (identifier, "ppm raw");    break;
  }  /* end switch */

  if (ncattget (ncId, dataId, PipsFileTypeAtt, identifier) == -1)
    ncattput (ncId, dataId, PipsFileTypeAtt, NC_CHAR, strlen (identifier)+1,
              (void*) identifier);

  fillValue = 0;
  if (ncattget (ncId, dataId, _FillValue, &fillValue) == -1)
    ncattput (ncId, dataId, _FillValue, *dataType, 1, (void*) &fillValue);

/*
 * Leave define mode, enter data mode
 */
  ncendef (ncId);         /* leave define mode */

  return (dataId);
}  /* end pipsVarDef */


/* end pipsVarDef.c */
